package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.xlplatform.coc.service.SCMTraceabilityService
import com.xebialabs.xlrelease.api.internal.ReleaseResource
import com.xebialabs.xlrelease.api.v1.forms.StartRelease
import com.xebialabs.xlrelease.api.v1.{FolderApi, ReleaseApi, RiskApi, TemplateApi}
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.service.{FolderService, ReleaseService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.Try


@Service
class ReleaseAsCodeService @Autowired()(
                                         templateApi: TemplateApi,
                                         folderService: FolderService,
                                         folderApi: FolderApi,
                                         releaseService: ReleaseService,
                                         releaseResource: ReleaseResource,
                                         folderAsCodeService: FolderAsCodeService,
                                         releaseApi: ReleaseApi,
                                         riskApi: RiskApi,
                                         scmTraceabilityService: SCMTraceabilityService
                                       ) {

  def startRelease(templatePath: String, releasePath: String, variables: Map[String, AnyRef], autoStart: Boolean, riskProfile: Option[String])(implicit metadata: Map[String, String]): Release = {
    val startRelease = new StartRelease
    val templateTitle = Ids.getName(templatePath)
    val releaseTitle = Ids.getName(releasePath)
    val templateFolder = Try(folderAsCodeService.searchParentFolder(templatePath, metadata)).getOrElse(None)
    val releaseFolderId = Try(folderAsCodeService.searchParentFolder(releasePath, metadata)).getOrElse(None)
    val profile = riskProfile.map(riskApi.getRiskProfileByTitle)

    val templatesInFolder = Utils.filterCisByFolder(
      releaseService.findTemplatesByTitle(templateFolder.map(_.getId).getOrElse(Ids.SEPARATOR), templateTitle, 0, 2, 1).asScala.toList,
      templateFolder.map(_.getId).getOrElse(Ids.SEPARATOR)
    )

    templatesInFolder match {
      case template :: Nil =>
        startRelease.setFolderId(releaseFolderId.map(_.getId).orNull)
        startRelease.setVariables(variables.asJava)
        startRelease.setReleaseTitle(releaseTitle)
        val createdRelease = templateApi.create(template.getId, startRelease)
        val scmDataId = Option(createdRelease.get$ciAttributes().getScmTraceabilityDataId)
        val scmData = scmDataId.flatMap(scmTraceabilityService.getById)
        profile.foreach(createdRelease.setProperty("riskProfile", _))
        val updatedRelease = releaseApi.updateRelease(createdRelease)
        scmData.foreach(releaseService.createSCMData(updatedRelease.getId, _))

        if (autoStart) {
          releaseApi.start(updatedRelease.getId)
        } else {
          updatedRelease
        }
      case Nil => throw new AsCodeException(s"No templates with the name [$templatePath] were found.")
      case _ => throw new ItemConflictException(s"There is more than one template with the name [$templatePath]. Please introduce an unique name.")
    }
  }
}
