package com.xebialabs.xlrelease.versioning.scheduler

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.QUARTZ_SCHEDULER_NAME
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.quartz.release.scheduler.ReleaseSchedulerService
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.versioning.scheduler.FolderVersioningAutoApplyJobService._
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDataMap, JobDetail, JobKey, SchedulerException}
import org.springframework.context.event
import org.springframework.core.annotation.Order
import org.springframework.retry.RetryCallback
import org.springframework.retry.support.RetryTemplateBuilder
import org.springframework.stereotype.Service

import java.time.Instant
import java.time.temporal.ChronoUnit.MINUTES
import java.util.Date
import java.util.concurrent.{ThreadLocalRandom, TimeUnit}
import scala.concurrent.duration.FiniteDuration
import scala.jdk.CollectionConverters._

@Service
class FolderVersioningAutoApplyJobService(releaseSchedulerService: ReleaseSchedulerService) extends Logging {

  private var running: Boolean = false

  private val retryTemplate = new RetryTemplateBuilder().fixedBackoff(250).maxAttempts(8).build()

  private val folderVersioningSettingsType = Type.valueOf(classOf[FolderVersioningSettings]).toString

  def handleAutoApplyGitVersion(folderVersioningSettings: FolderVersioningSettings): Unit = {
    if (folderVersioningSettings.getAutoImport) {
      schedule(folderVersioningSettings)
    } else {
      unschedule(folderVersioningSettings.getFolderId)
    }
  }

  def unschedule(folderId: String): Unit = {
    if (running) {
      val jobKey = new JobKey(Ids.getName(folderId), folderVersioningSettingsType)
      val retryCallback: RetryCallback[Boolean, SchedulerException] = _ => releaseSchedulerService.unschedule(jobKey)
      retryTemplate.execute(retryCallback)
      logger.debug(s"Unscheduled a job associated with folder [$folderId] to auto apply new changes with folder versioning")
    } else {
      logger.debug(s"Scheduler is not started yet. Cannot unschedule a quartz job.")
    }
  }

  private def schedule(folderVersioningSettings: FolderVersioningSettings): Unit = {
    if (running) {
      val identity = Ids.getName(folderVersioningSettings.getFolderId)
      val duration = FiniteDuration.apply(folderVersioningSettings.getPollDuration, TimeUnit.MINUTES)

      val job: JobDetail = newJob(classOf[FolderVersioningAutoApplyQuartzJob])
        .withDescription(s"Job to auto apply new changes with folder versioning")
        .withIdentity(identity, folderVersioningSettingsType)
        .storeDurably(true)
        .setJobData(new JobDataMap(Map[String, String](AUTO_APPLY_JOB_DATA_KEY -> folderVersioningSettings.getFolderId).asJava))
        .build()

      val quartzTrigger = newTrigger()
        .withDescription(s"Trigger to auto apply new changes with folder versioning")
        .withIdentity(identity, folderVersioningSettingsType)
        .withSchedule(
          simpleSchedule()
            .withIntervalInMilliseconds(duration.toMillis)
            .repeatForever()
            .withMisfireHandlingInstructionNextWithRemainingCount() // In case of misfire, trigger scheduled time should not change
        )
        .startAt(getRandomTriggerStartDate(duration))
        .build()

      releaseSchedulerService.schedule(job, quartzTrigger)

      logger.debug(s"Scheduled a job associated with folder [$identity] to auto apply new changes with folder versioning")
    } else {
      logger.debug(s"Scheduler is not started yet. Cannot schedule a quartz job.")
    }
  }

  /**
   * Returns random start date for trigger depending upon duration
   * This is to avoid potential having the same start time for multiple triggers and firing at the same time
   */
  private def getRandomTriggerStartDate(duration: FiniteDuration): Date = {
    val durationInMins = duration.toMinutes
    val now = Instant.now().plus(durationInMins, MINUTES).toEpochMilli
    val end = Instant.now().plus(durationInMins * 2, MINUTES).toEpochMilli
    val randomMillisSinceEpoch = ThreadLocalRandom.current().nextLong(now, end)
    val randomInstant = Instant.ofEpochMilli(randomMillisSinceEpoch)
    Date.from(randomInstant)
  }

  @event.EventListener
  @Order(1) // To make sure running = true is set before ScheduleDefaultContentFolderAutoApplyService listener is called
  def onStartup(event: SchedulerStartedEvent): Unit = {
    if (null != event && null != event.schedulerName && event.schedulerName == QUARTZ_SCHEDULER_NAME) {
      running = true
    }
  }
}

object FolderVersioningAutoApplyJobService {
  val AUTO_APPLY_JOB_DATA_KEY: String = "folderId"
}
