package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.domain.events.{FolderVariableCreatedEvent, FolderVariableUpdatedEvent}
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.ValidationMessage
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.FolderVariableRepository
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

@Service
class FolderVariableAsCodeService @Autowired()(folderVariableRepository: FolderVariableRepository,
                                               referenceSolver: ReferenceSolver,
                                               eventBus: XLReleaseEventBus) extends Logging {

  def process(context: ImportContext, variable: Variable): ImportResult = {
    logger.debug(s"Processing folder variable: ${variable.toString} with metadata ${context.metadata.toString}")

    variable.setFolderId(context.scope.getFolderId.get)

    referenceSolver.resolveReferences(variable, context.references, variable.getFolderId)
    referenceSolver.resolveStringReference(variable, variable.getFolderId, context.scope.getMetadataHome)

    find(context, variable) match {
      case Some(existing) =>
        logger.debug(s"Updating folder variable: ${existing.toString}")
        update(context, existing, variable)
      case None =>
        logger.debug(s"Creating folder variable: ${variable.toString}")
        create(context, variable)
    }
  }

  private def find(context: ImportContext, variable: Variable): Option[Variable] = {
    fixUpPrefix(variable, "folder")
    try {
      Some(folderVariableRepository.findByKey(variable.getKey, context.scope.getFolderId.orNull))
    } catch {
      case _: Throwable => None
    }
  }

  private def create(context: ImportContext, variable: Variable): ImportResult = {
    // DB layer handles all the id creation and initialization
    val messages = validate(context, variable)
    val created = folderVariableRepository.create(variable, context.scope.getFolderCiUid.orNull)
    ImportResult(
      List(CI.ids.withCreated(created.getId)),
      Seq(() => eventBus.publish(FolderVariableCreatedEvent(created, null))),
      Map.empty,
      messages.map(msg => ValidationMessage.updateCiId(msg, created.getId())) // need to insert ids
    )
  }

  private def update(context: ImportContext, existing: Variable, variable: Variable): ImportResult = {
    // DB layer handles all the initialization
    checkSameType(variable, existing)
    variable.setId(existing.getId)
    val messages = validate(context, variable)
    val updated = folderVariableRepository.update(variable, context.scope.getFolderCiUid.orNull)
    ImportResult(
      List(CI.ids.withUpdated(updated.getId)),
      Seq(() => eventBus.publish(FolderVariableUpdatedEvent(existing, updated, null))),
      Map.empty,
      messages
    )
  }

  private def validate(context: ImportContext, variable: Variable): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(variable, context.getFolderInfo()).toList
      case None => List.empty
    }
  }
}
