package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.Version.XLR_API_VERSION
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, GenerateService}
import com.xebialabs.xlrelease.ascode.utils.StaticVariables.XLR_TEMPLATE_KIND
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.GENERATE_FOLDER_CONFIGURATION
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
class DefinitionsGenerator @Autowired()(folderAsCodeService: FolderAsCodeService,
                                        generateService: GenerateService,
                                        permissions: PermissionChecker) {

  def generate(folderId: String, config: FolderVersioningSettings): List[Definition] = {
    val cisConfig = CisConfig(
      generateConfigurations = config.getExportConfiguration,
      generateDashboards = config.getExportDashboards,
      generateTemplates = config.getExportTemplates,
      generateWorkflows = config.getExportWorkflows,
      generateTriggers = config.getExportTriggers,
      generateVariables = config.getExportVariables,
      generateDeliveryPatterns = config.getExportPatterns,
      generateNotificationSettings = config.getExportNotifications,
      generatePermissions = config.getExportSecurity,
      generateApplications = config.getExportApplications,
      generateEnvironments = config.getExportEnvironments
    )

    try {
      validateGeneratePermissions(folderId)
      val scope = FolderSearch(folderAsCodeService.getFolderPath(folderId), folderId)
      val excludeIds = Seq(config.getGitConnection.getId)
      generateService.generate(GeneratorConfig(None, scope, cisConfig, permissions.hasGlobalPermission(ADMIN), excludedEntities = excludeIds))
    } catch {
      case e: AsCodeException =>
        e.getMessage match {
          case GenerateService.EMPTY_DEFINITION_ERROR =>
            List(Definition(XLR_API_VERSION,
              Option(Map(MetadataFields.HOMEFOLDER.toString -> FolderSearch(folderAsCodeService.getFolderPath(folderId), folderId).path)),
              XLR_TEMPLATE_KIND,
              CiSpec(List.empty, List.empty, Map.empty)))
          case _ => throw ScmException(e.getMessage, e.getCause)
        }
    }
  }

  private def validateGeneratePermissions(folderId: String): Unit = {
    if (!(permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId) || permissions.isCurrentUserAdmin)) {
      throw ScmException(s"Unable to generate configuration: must have Admin or 'Generate Configuration' permission on the folder")
    }
  }
}
