package com.xebialabs.xlrelease.versioning.ascode.repository

import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.versioning.ascode.form.{ExternalDependencyReportMessage, ValidationReportMessage}
import com.xebialabs.xlrelease.versioning.ascode.repository.sql.persistence.{FolderValidationMessagePersistence, FolderValidationMessageRow}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository

@IsTransactional
@Repository
@Autowired
class SqlFolderValidationMessageRepository(folderValidationMessagePersistence: FolderValidationMessagePersistence
                                          ) extends FolderValidationMessageRepository with Logging {

  @Timed
  @IsReadOnly
  override def findValidationMessageById(messageId: CiUid): Option[ValidationReportMessage] = {
    logger.debug(s"Getting folder validation message with id [$messageId]")
    folderValidationMessagePersistence.findById(messageId).map(folderValidationMessageRowToReportMessage)
  }

  @Timed
  @IsReadOnly
  override def findValidationMessagesByFolderId(folderId: String): Seq[ValidationReportMessage] = {
    logger.debug(s"Getting validation message entries for folderId [$folderId]")
    folderValidationMessagePersistence.findByFolderId(folderId).map(folderValidationMessageRowToReportMessage)
  }

  @Timed
  override def updateValidationMessageResolvedState(messageId: CiUid, validationMessage: ValidationReportMessage): Unit = {
    logger.debug(s"Updating folder validation message with id [$messageId] with resolvedStatus [$validationMessage.resolvedStatus]")
    folderValidationMessagePersistence.update(messageId, validationMessage)
  }

  @Timed
  override def createValidationMessageEntity(folderId: String, validationMessage: ValidationReportMessage): Unit = {
    logger.debug(s"Creating validation message entry for folderId [$folderId] with resolvedStatus [${validationMessage.resolvedStatus}] and message [$validationMessage]")
    folderValidationMessagePersistence.insert(folderId, validationMessage)
  }

  @Timed
  override def deleteValidationMessage(messageId: CiUid): Unit = {
    logger.debug(s"Deleting folder validation message with id [$messageId]")
    folderValidationMessagePersistence.delete(messageId)
  }

  private def folderValidationMessageRowToReportMessage(row: FolderValidationMessageRow): ExternalDependencyReportMessage = {
    val message = new ExternalDependencyReportMessage
    message.id = row.ciInfoId
    message.ciType = row.ciInfoType
    message.ciTypeDisplayName = row.ciInfoDisplayName
    message.title = row.ciInfoTitle
    message.folder = row.ciInfoFolderPath
    message.folderId = row.ciInfoFolderId
    message.property = row.property
    message.message = row.message
    message.messageType = row.messageType
    message.messageId = row.messageId
    message.resolvedStatus = row.resolvedStatus
    message.dependency = row.dependency
    message
  }

}
