package com.xebialabs.xlrelease.ascode.utils

import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.versioning.ascode.{FolderInfo, ImportValidator}
import com.xebialabs.xltype.serialization.CiReference

import scala.annotation.tailrec

sealed trait ImportScope {
  def description(): String = {
    this match {
      case GlobalScope => "in global scope"
      case f: FolderScope => s"in folder ${f.path}"
      case t: TemplateScope => s"in template ${t.title} ${t.scope.description()}"
      case _ => "unknown"
    }
  }

  @tailrec
  final def getFolderId: Option[String] = {
    this match {
      case GlobalScope => None
      case f: FolderScope => Some(f.id)
      case t: TemplateScope => t.scope.getFolderId
      case _ => None
    }
  }

  @tailrec
  final def getFolderPath: Option[String] = {
    this match {
      case GlobalScope => None
      case f: FolderScope => Some(f.path)
      case t: TemplateScope => t.scope.getFolderPath
      case _ => None
    }
  }

  @tailrec
  final def getFolderCiUid: Option[CiUid] = {
    this match {
      case GlobalScope => None
      case f: FolderScope => Some(f.ciUid)
      case t: TemplateScope => t.scope.getFolderCiUid
      case _ => None
    }
  }

  @tailrec
  final def getFolderSecurityUid: Option[CiUid] = {
    this match {
      case GlobalScope => None
      case f: FolderScope => Some(f.securityUid)
      case t: TemplateScope => t.scope.getFolderSecurityUid
      case _ => None
    }
  }

  @tailrec
  final def getMetadataHome: Option[String] = {
    this match {
      case GlobalScope => None
      case f: FolderScope => f.home
      case t: TemplateScope => t.scope.getMetadataHome
      case _ => None
    }
  }
}

case object GlobalScope extends ImportScope

case class FolderScope(id: String, path: String, ciUid: CiUid, securityUid: CiUid, home: Option[String]) extends ImportScope

case class TemplateScope(id: String, title: String, scope: ImportScope) extends ImportScope

case class ImportContext(scope: ImportScope,
                         metadata: Map[String, String] = Map.empty,
                         references: List[CiReference] = List.empty,
                         scmData: Option[SCMTraceabilityData] = None,
                         templateIds: Map[String, (String, Release)] = Map.empty,
                         validator: Option[ImportValidator] = None) {
  def getFolderInfo(): FolderInfo = {
    val relative = scope.getFolderPath match {
      case Some(path) => {
        scope.getMetadataHome match {
          case Some(home) =>
            if (path.equals(home)) {
              ""
            }
            else if (path.startsWith(s"${home}/")) {
              path.substring(home.length + 1)
            }
            else {
              path
            }
          case _ => path
        }
      }
      case None => null
    }

    FolderInfo(
      scope.getFolderId.orNull,
      relative,
      scope.getFolderPath.orNull
    )
  }
}
