package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds.{CI, _}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.ascode.service.spec.CiSpecInterpreter.ProcessedCi
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.domain.events.{ConfigurationCreatedEvent, ConfigurationUpdatedEvent}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings.GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID
import com.xebialabs.xlrelease.notifications.email.EmailNotificationsService
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.Ids.MAIL_CONFIGURATION_ROOT
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters.CollectionHasAsScala

@Service
class NotificationSettingsAsCodeService(configurationRepository: ConfigurationRepository,
                                        ciIdService: CiIdService,
                                        emailNotificationsService: EmailNotificationsService,
                                        eventBus: XLReleaseEventBus) extends Logging {

  def process(context: ImportContext, settings: EmailNotificationSettings): ImportResult = {
    val processed = find(context, settings) match {
      case Some(existing) =>
        val updated = update(context, existing, settings)
        ProcessedCi(updated, CI.ids.withUpdated(updated.getId), Seq(() => eventBus.publish(ConfigurationUpdatedEvent(updated))))
      case None =>
        val created = create(context, settings)
        ProcessedCi(created, CI.ids.withCreated(created.getId), Seq(() => eventBus.publish(ConfigurationCreatedEvent(created))))
    }

    ImportResult(List(processed.changedIds), processed.postCommitActions)
  }

  private def find(context: ImportContext, settings: EmailNotificationSettings): Option[EmailNotificationSettings] = {
    context.scope.getFolderId match {
      case Some(folderId) => configurationRepository
        .findAllByTypeAndTitle(Type.valueOf(classOf[EmailNotificationSettings]), null, folderId, folderOnly = true).asScala.headOption
      case None => Some(configurationRepository.read(GLOBAL_EMAIL_NOTIFICATION_SETTINGS_ID).asInstanceOf[EmailNotificationSettings])
    }
  }

  private def create(context: ImportContext, settings: EmailNotificationSettings): EmailNotificationSettings = {
    settings.setId(ciIdService.getUniqueId(Type.valueOf(classOf[EmailNotificationSettings]), MAIL_CONFIGURATION_ROOT))
    settings.setFolderId(context.scope.getFolderId.orNull)
    configurationRepository.create(settings, context.scope.getFolderCiUid.orNull)
  }

  private def update(context: ImportContext, existing: EmailNotificationSettings, settings: EmailNotificationSettings): EmailNotificationSettings = {
    settings.setId(existing.getId)
    settings.setFolderId(context.scope.getFolderId.orNull)
    if (settings.getFolderId == null) {
      // Only relevant for global notifications
      emailNotificationsService.validateStandaloneSettings(settings)
    }
    configurationRepository.update(settings, context.scope.getFolderCiUid.orNull)
  }
}
