package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.ascode.yaml.sugar.GenerateStrategy.Reject
import com.xebialabs.ascode.yaml.sugar.Sugarizer._
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.service.referencehandler.cireferencehandler.{CiReferenceCreationConfig, CiReferenceCreationService}
import com.xebialabs.xlrelease.ascode.service.referencehandler.generate.{CiReferenceGeneratorService, CiWithHome}
import com.xebialabs.xlrelease.ascode.service.referencehandler.stringhandler.{StringReferenceCreationConfig, StringReferenceCreationService}
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.yaml.sugar.XLRSugar._
import com.xebialabs.xlrelease.domain.TemplateLogo
import com.xebialabs.xlrelease.domain.scm.connector.ScmCredential
import com.xebialabs.xlrelease.domain.variables.ExternalVariableValue
import com.xebialabs.xlrelease.notifications.configuration.SmtpAuthentication
import com.xebialabs.xlrelease.triggers.event_based.{ExpressionEventFilter, GroovyEventFilter, JythonEventFilter, ScriptBasedEventFilter}
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.webhooks.authentication.BaseRequestAuthentication
import com.xebialabs.xltype.serialization.CiReference
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.Try

@Service
class ReferenceSolver @Autowired()(
                                    ciReferenceGeneratorService: CiReferenceGeneratorService,
                                    ciReferenceCreationService: CiReferenceCreationService,
                                    stringReferenceCreationService: StringReferenceCreationService
                                  ) extends Logging {

  private lazy val classesWithoutTitle = List(
    typeOf[ScmCredential],
    typeOf[SmtpAuthentication],
    typeOf[BaseRequestAuthentication],
    typeOf[ScriptBasedEventFilter],
    typeOf[JythonEventFilter],
    typeOf[GroovyEventFilter],
    typeOf[ExpressionEventFilter],
    typeOf[ExternalVariableValue],
    typeOf[TemplateLogo]
  )

  private def referencesPropertyFilter(property: PropertyDescriptor): Boolean = {
    val isCi = property.getKind == PropertyKind.CI
    val isCollectionOfCi = property.getKind == PropertyKind.SET_OF_CI || property.getKind == PropertyKind.LIST_OF_CI
    val isBlacklisted = isCi && property.getReferencedType.getClosestTypeSugar.exists(_.generateHints.strategy == Reject)
    val isPythonScript = isCi && Utils.isTypeAPythonScript(property.getReferencedType)
    val isNested = isCi && property.isNested

    (isCi || isCollectionOfCi) && !isBlacklisted && !isPythonScript && !isNested
  }

  def obtainReferencesFromCi(ci: ConfigurationItem, home: Option[String]): List[CiReference] = {
    // TODO move this somewhere else
    logger.debug(s"Obtaining references from ci [${ci.getId}]")
    ciReferenceGeneratorService.typeSpecificReferenceGenerate(CiWithHome(ci, home))

    ci.getType
      .getDescriptor
      .getPropertyDescriptors
      .asScala
      .filter(referencesPropertyFilter)
      .flatMap { property =>
        Option(property.get(ci)).toList.flatMap {
          case collection: java.util.Collection[_]  =>
            collection.asScala.toList.flatMap {
              case referencedCi: ConfigurationItem =>
                Try(createCiReference(referencedCi, property)) match {
                  case scala.util.Success(ciReference) => Option(ciReference)
                  case scala.util.Failure(_: AsCodeException) =>
                    logger.debug(s"Failed to create reference for ci [${referencedCi.getId}] with property [${property.getName}] " +
                      s"- type is not marked as titleless")
                    None
                  case scala.util.Failure(e) =>
                    throw e
                }
            }
          case referencedCi: ConfigurationItem =>
            List(createCiReference(referencedCi, property))
          case _ => List.empty
        }
      }.toList
  }

  private def createCiReference(referencedCi: ConfigurationItem, property: PropertyDescriptor): CiReference = {
    val referenceTitle: String = Utils.getCiTitle(referencedCi)
      .getOrElse {
        if (classesWithoutTitle.exists(referencedCi.getType.instanceOf)) {
          ""
        } else {
          throw new AsCodeException(s"The referenced ci with type [${referencedCi.getType.toString}] is not supported yet.")
        }
      }
    new CiReference(referencedCi, property, referenceTitle)
  }

  // TODO: templateIds should be a more generic cache lookup mechanism but the map is ok for the time being
  def resolveStringReference(ci: ConfigurationItem, parentFolder: String, home: Option[String] = None, templateIds: Map[String, String] = Map.empty): Unit = {
    stringReferenceCreationService.createStringReference(
      StringReferenceCreationConfig(
        ci, parentFolder, home, templateIds
      )
    )
  }

  def resolveReference(ci: ConfigurationItem,
                       propertyDescriptor: PropertyDescriptor,
                       folderPath: String,
                       reference: CiReference): Unit = {
    ciReferenceCreationService.createTypeSpecificReference(CiReferenceCreationConfig(
      ci,
      propertyDescriptor,
      folderPath,
      reference
    ))
  }

  def resolveReferences(ciToResolve: ConfigurationItem, references: List[CiReference], folderPath: String): Unit = {
    logger.debug(s"Resolving references for object of type: ${ciToResolve.getType.toString}")
    val nestedCis = CiHelper.getNestedCis(ciToResolve).asScala.toList

    references.filter(reference => nestedCis.contains(reference.getCi)).foreach { reference =>
      val ci = reference.getCi
      val propertyDescriptor = reference.getProperty

      resolveReference(ci, propertyDescriptor, folderPath, reference)
    }
  }
}
