package com.xebialabs.xlrelease.ascode.service.spec

import com.xebialabs.ascode.service.spec.SpecInterpreter._
import com.xebialabs.ascode.service.spec.{InterpreterContext, SpecInterpreter}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.xlrelease.api.v1.{EnvironmentApi, EnvironmentLabelApi, EnvironmentReservationApi, EnvironmentStageApi}
import com.xebialabs.xlrelease.ascode.service.spec.XLREntityKinds._
import com.xebialabs.xlrelease.ascode.utils.DateUtils
import com.xebialabs.xlrelease.ascode.yaml.model.{Environment, EnvironmentsSpec, Label, Reservation}
import com.xebialabs.xlrelease.builder.EnvironmentBuilder
import com.xebialabs.xlrelease.domain.environments.{EnvironmentLabel, EnvironmentReservation, EnvironmentStage, Environment => ApiEnvironment}
import com.xebialabs.xlrelease.environments.repository._
import com.xebialabs.xlrelease.environments.service.EnvironmentReservationSecurity
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.Date
import scala.jdk.CollectionConverters._

@Component
@Autowired
class EnvironmentsSpecInterpreter(
                                   environmentApi: EnvironmentApi,
                                   stageApi: EnvironmentStageApi,
                                   labelApi: EnvironmentLabelApi,
                                   environmentReservationApi: EnvironmentReservationApi,
                                   environmentRepository: EnvironmentRepository,
                                   stageRepository: EnvironmentStageRepository,
                                   labelRepository: EnvironmentLabelRepository,
                                   applicationRepository: ApplicationRepository,
                                   environmentReservationRepository: EnvironmentReservationRepository,
                                   environmentReservationSecurity: EnvironmentReservationSecurity
                                 ) extends SpecInterpreter with Logging {

  override def isDefinedAt(context: InterpreterContext): Boolean = context.definition.spec.isInstanceOf[EnvironmentsSpec]

  private def applyStages(stages: List[String]): ChangedIds = {
    createOrUpdate[String, EnvironmentStage](
      entities = stages,
      defaultIds = STAGE.ids,
      extractLabel = stage => stage.getTitle,
      prepare = title => {
        val stage = new EnvironmentStage
        stage.setTitle(title)
        stage
      },
      create = stageApi.create,
      update = stage => {
        val loadedStage = stageRepository.findByTitle(stage.getTitle)
        stage.setId(loadedStage.getId)
        stageApi.update(stage)
      }
    )
  }

  private def applyLabels(labels: List[Label]): ChangedIds = {
    createOrUpdate[Label, EnvironmentLabel](
      entities = labels,
      defaultIds = LABEL.ids,
      extractLabel = label => label.getTitle,
      prepare = label => {
        val environmentLabel = new EnvironmentLabel
        environmentLabel.setTitle(label.name)
        environmentLabel.setColor(label.color)
        environmentLabel
      },
      create = labelApi.create,
      update = environmentLabel => {
        val loadedLabel = labelRepository.findByTitle(environmentLabel.getTitle)
        environmentLabel.setId(loadedLabel.getId)
        labelApi.update(environmentLabel)
      }
    )
  }

  private def applyEnvironments(environments: List[Environment]): ChangedIds = {
    createOrUpdate[Environment, ApiEnvironment](
      entities = environments,
      defaultIds = ENVIRONMENT.ids,
      extractLabel = environment => environment.getTitle,
      prepare = environment => {
        EnvironmentBuilder.newEnvironment
          .withTitle(environment.name)
          .withStage(stageRepository.findByTitle(environment.stage))
          .withLabels(environment.labels.map(labelRepository.findByTitle).asJava)
          .withDescription(environment.description.orNull)
          .build
      },
      create = environmentApi.create,
      update = apiEnvironment => {
        val loadedEnvironment = environmentRepository.findEnvironmentByTitle(apiEnvironment.getTitle)
        apiEnvironment.setId(loadedEnvironment.getId)
        environmentApi.update(apiEnvironment)
      }
    )
  }

  private def applyReservations(reservations: List[Reservation]): ChangedIds = {
    if (reservations.nonEmpty) {
      environmentReservationSecurity.checkCreate()
      val now = new Date()

      reservations
        .groupBy(_.environment)
        .foldLeft(CI.ids) { case (ids, (envTitle, newReservations)) =>
          val environment = environmentRepository.findEnvironmentByTitle(envTitle)

          // create new future reservations
          val createdIds = newReservations.map { r =>
            val environmentReservation = new EnvironmentReservation()

            environmentReservation.setEnvironment(environment)
            environmentReservation.setApplications(
              r.applications.map(applicationRepository.findApplicationByTitle).asJava
            )

            environmentReservation.setStartDate(DateUtils.fromString(r.startDate))
            environmentReservation.setEndDate(DateUtils.fromString(r.endDate))
            environmentReservation.setNote(r.note)
            environmentReservationApi.create(environmentReservation).getId
          }

          ids.withCreated(createdIds)
        }
    } else {
      CI.ids
    }
  }

  override def apply(context: InterpreterContext): AsCodeResponse = {
    val spec = context.definition.spec.asInstanceOf[EnvironmentsSpec]
    val stageIds = applyStages(spec.stages)
    val labelIds = applyLabels(spec.labels)
    val environmentIds = applyEnvironments(spec.environments)
    val reservationIds = applyReservations(spec.reservations)

    AsCodeResponse.ids(List(
      stageIds.toOption,
      labelIds.toOption,
      environmentIds.toOption,
      reservationIds.toOption
    ).flatten: _*)
  }
}
