package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.ascode.service.spec.CiSpecInterpreter.ProcessedCi
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.domain.events.{GlobalVariableCreatedEvent, GlobalVariableUpdatedEvent}
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class GlobalVariableAsCodeService @Autowired()(configurationRepository: ConfigurationRepository,
                                               referenceSolver: ReferenceSolver,
                                               ciIdService: CiIdService,
                                               eventBus: XLReleaseEventBus) extends Logging {

  def process(context: ImportContext, variable: Variable): ImportResult = {
    logger.debug(s"Processing global variable: ${variable.toString} with metadata ${context.metadata.toString}")

    referenceSolver.resolveReferences(variable, context.references, null)

    val processed = find(context, variable) match {
      case Some(existing) =>
        logger.debug(s"Updating global variable: ${existing.toString}")
        val updated = update(context, existing, variable)
        ProcessedCi(updated, CI.ids.withUpdated(updated.getId), Seq(() => eventBus.publish(GlobalVariableUpdatedEvent(existing, updated, null))))
      case None =>
        logger.debug(s"Creating global variable: ${variable.toString}")
        val created = create(context, variable)
        ProcessedCi(created, CI.ids.withCreated(created.getId), Seq(() => eventBus.publish(GlobalVariableCreatedEvent(created, null))))
    }

    ImportResult(List(processed.changedIds), processed.postCommitActions)
  }

  private def find(context: ImportContext, variable: Variable): Option[Variable] = {
    fixUpPrefix(variable, "global")
    val title = variable.getKey

    val matches = configurationRepository
      .findAllByType[Variable](Type.valueOf(classOf[Variable])).asScala.toList
      .filter(v => v.getKey.equals(title)) // TODO: Dumb but global variables can't be queried by name...should prelookup and pass in through the import context

    if (matches.length > 1) {
      logger.debug(s"There are multiple global variables named [${title}] ${context.scope.description()}, those are: ${matches.toString}")
      throw new AsCodeException(s"Multiple global variables are named [${title}] ${context.scope.description()}. Can not determine which one to update.")
    }

    matches.headOption
  }

  private def create(context: ImportContext, variable: Variable): Variable = {
    fixUpGlobalVariable(variable, ciIdService)
    configurationRepository.create(variable)
  }

  private def update(context: ImportContext, existing: Variable, variable: Variable): Variable = {
    checkSameType(variable, existing)
    variable.setId(existing.getId)
    fixUpGlobalVariable(variable, ciIdService)
    configurationRepository.update(variable)
  }
}
