package com.xebialabs.xlrelease.ascode.utils

import com.xebialabs.ascode.exception.{AsCodeException, CiValidationException}
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.ascode.utils.Utils._
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.CiValidationError
import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Property.Size.LARGE
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.api.v1.forms
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.generatestrategy.CiGenerateConfig
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xltype.serialization.CiReference
import org.apache.commons.lang.StringEscapeUtils.unescapeJava

import java.util.regex.Pattern
import scala.collection.immutable.ListMap
import scala.jdk.CollectionConverters._

case class FolderPathAndTitle(folderPath: Option[String], title: String) {
  def replaceRelativePath(parent: Option[String]): FolderPathAndTitle = {
    folderPath match {
      case Some(path) if (path.startsWith(".") && parent.isDefined) => {
        FolderPathAndTitle(Some(path.replaceFirst(".", parent.get)), title)
      }
      case _ => this
    }
  }

  def absolutePath(): String = {
    folderPath match {
      case None => title
      case Some(path) => Utils.joinPaths(Seq(path, title))
    }
  }
}

object Utils {
  def containsUnescapedPath(title: String): Boolean = {
    val regex = "(?<!\\\\)/".r
    regex.findFirstIn(title).isDefined
  }

  def removeEmptyHead(splitTarget: Array[String]): Array[String] = {
    splitTarget.headOption match {
      case Some("") => splitTarget.tail
      case _ => splitTarget
    }
  }


  private def isDefaultValueOrEmpty(ci: ConfigurationItem, descriptor: PropertyDescriptor): Boolean = {
    isValueEmpty(ci, descriptor) || isDefaultValue(ci, descriptor)
  }

  private def isDefaultValue(ci: ConfigurationItem, descriptor: PropertyDescriptor): Boolean = {
    val value = descriptor.get(ci)
    val default = descriptor.getDefaultValue

    value == default
  }

  private def isValueEmpty(ci: ConfigurationItem, descriptor: PropertyDescriptor): Boolean = {
    val value = descriptor.get(ci)

    value == null
  }

  def getCiTitle(ci: ConfigurationItem): Option[String] = {
    ci.getType.getDescriptor.getPropertyDescriptor("title") match {
      case null =>
        ci match {
          case ciWithoutTitle if ciWithoutTitle.isInstanceOf[ValueProviderConfiguration] => Some(ciWithoutTitle.asInstanceOf[ValueProviderConfiguration].getVariable.getKey)
          case ciWithoutTitle if ciWithoutTitle.isInstanceOf[Variable] => Some(ciWithoutTitle.asInstanceOf[Variable].getKey)
          case ciWithoutTitle if ciWithoutTitle.isInstanceOf[Dependency] => Some(ciWithoutTitle.asInstanceOf[Dependency].getTargetId)
          case ciWithoutTitle if isTypeAPythonScript(ciWithoutTitle) => Some(ciWithoutTitle.asInstanceOf[PythonScript].getCustomScriptTask.getTitle)
          case _ => None
        }
      case descriptor if ci.isInstanceOf[Dashboard] =>
        val title = if (isDefaultValueOrEmpty(ci, descriptor)) None else Some(descriptor.get(ci).asInstanceOf[String])
        val parentIdDescriptor = ci.getType.getDescriptor.getPropertyDescriptor("parentId")
        val parentTitle = if (isValueEmpty(ci, parentIdDescriptor)) None else Some(parentIdDescriptor.get(ci).asInstanceOf[String])

        if (title.isDefined && parentTitle.isDefined) {
          throw new AsCodeException(s"The ci with type [${ci.getType.toString}] have the fields [title, parentId] filled, please just fill one field.")
        }

        if (title.isEmpty && !isDefaultValue(ci, descriptor) && parentTitle.isEmpty) {
          throw new AsCodeException(s"The ci with type [${ci.getType.toString}] have the fields [title, parentId] empty.")
        }

        Some(title.getOrElse(parentTitle.getOrElse(descriptor.get(ci).asInstanceOf[String])))
      case descriptor =>
        val title = descriptor.get(ci)

        if (title == null) {
          throw new AsCodeException(s"The ci with type [${ci.getType.toString}] have the field [title] empty.")
        } else {
          Some(title.asInstanceOf[String])
        }
    }
  }

  def getCiPathAndSubstitute(ci: ConfigurationItem): String = {
    val propertyDescriptor = Option(ci.getType.getDescriptor.getPropertyDescriptor("title"))

    propertyDescriptor.map { descriptor =>
      val ciTitle = Option(descriptor.get(ci).asInstanceOf[String])
      val relativePath = getFolderPathFromCiPath(ciTitle.getOrElse("")).getOrElse("")

      descriptor.set(ci, getName(ciTitle.orNull).orNull)
      relativePath
    }.getOrElse("")
  }

  def getFolderPathFromCiPath(ciPath: String): Option[String] = {
    val splittedPath = splitStringByPathSeparator(ciPath.stripMargin('/'))

    if (splittedPath.length > 1) {
      Some(splittedPath.slice(0, splittedPath.length - 1).mkString(Ids.SEPARATOR))
    } else {
      None
    }
  }

  def checkIfTriggerVariablesAreCorrect(releaseVariables: List[Variable], triggerVariables: List[Variable], triggerTitle: String): Unit = {
    val nullValueVariables = triggerVariables.filter(_.getValue == null)

    if (nullValueVariables.nonEmpty)
      throw new AsCodeException(s"The variables [${nullValueVariables.map(_.getKey).mkString(", ")}] that were set in the trigger [$triggerTitle] got no value setup.")

    val missingVariables = triggerVariables.filterNot(trigVar =>
      releaseVariables.exists(relVal => relVal.getKey == trigVar.getKey && relVal.getShowOnReleaseStart)
    )

    if (missingVariables.nonEmpty)
      throw new AsCodeException(s"The variables [${missingVariables.map(_.getKey).mkString(", ")}] that were set in the trigger [$triggerTitle] are incorrect.")
  }

  def filterCisByFolder[T <: ConfigurationItem](cis: List[T], folderId: String = Ids.SEPARATOR): List[T] = {
    cis.filter { x =>
      if (folderId != Ids.SEPARATOR) {
        x.getId.startsWith(folderId)
      } else {
        Ids.isInRootFolder(x.getId)
      }
    }
  }

  private def isStringEmpty(value: AnyRef) = value.isInstanceOf[String] && value == ""

  def isRequiredReferenceAndDontHaveValue(referencesWithType: List[CiReference], field: PropertyDescriptor): Boolean = {
    !referencesWithType.exists(_.getProperty == field)
  }

  def checkRequiredCiFields[T <: ConfigurationItem](ci: T, references: List[CiReference] = List.empty): Unit = {
    val ciType = ci.getType
    val descriptor = ciType.getDescriptor
    val requiredFieldsWithoutReferences = descriptor.getPropertyDescriptors.asScala.filter(property => property.isRequired && property.getReferencedType == null).toList
    val requiredFieldsWithReferences = descriptor.getPropertyDescriptors.asScala.filter(property => property.isRequired && property.getReferencedType != null).toList
    val referencesWithType = references.filter(_.getCi.getType == ciType)

    val requiredFieldsWithoutReferencesEmpty = requiredFieldsWithoutReferences
      .withFilter(field => field.get(ci) == null || isStringEmpty(field.get(ci)))
      .map(_.getName)

    val requiredFieldsWithReferencesEmpty = requiredFieldsWithReferences
      .withFilter(field => isRequiredReferenceAndDontHaveValue(referencesWithType, field))
      .map(_.getName)

    val allMissingFields = requiredFieldsWithoutReferencesEmpty ::: requiredFieldsWithReferencesEmpty

    if (allMissingFields.nonEmpty) {
      throw CiValidationException(
        allMissingFields.map(field => CiValidationError(ci.getId, field, "The field is required but was not filled"))
      )
    }
  }

  def isTypeAPythonScript[T <: ConfigurationItem](ci: T): Boolean = {
    isTypeAPythonScript(ci.getType)
  }

  def isTypeAPythonScript(ciType: String): Boolean = {
    isTypeAPythonScript(typeOf(ciType))
  }

  def isTypeAPythonScript(ciType: Type): Boolean = {
    PythonScriptDefinition.isScriptDefinition(ciType)
  }

  def beginsWithCaret(string: String): Boolean = {
    string.startsWith("^")
  }

  def escapeWithCaret(name: String): String = s"^$name"

  def joinMapOfLists[X, Y](
                            map1: ListMap[Option[X], List[Y]],
                            map2: ListMap[Option[X], List[Y]]
                          ): ListMap[Option[X], List[Y]] = {
    map1 ++ map2.map {
      case (key, value) => key -> map1.get(key).map(_ ++ value).getOrElse(value)
    }
  }

  def buildFolderPath(parent: String, title: String): String = {
    val path = if (parent == Ids.SEPARATOR) {
      s"$parent$title"
    } else {
      s"$parent${Ids.SEPARATOR}$title"
    }

    if (path.startsWith(Ids.SEPARATOR)) {
      path.dropWhile(_ == '/')
    } else {
      path
    }
  }

  def buildDashboardIdFromTemplateId(templateId: String): String = {
    templateId + Ids.SEPARATOR + Dashboard.DASHBOARD_PREFIX
  }

  def isTypeOrSubtype(ciType: Type, comparativeType: String): Boolean = {
    typeOf(comparativeType).isSubTypeOf(ciType) || typeOf(comparativeType) == ciType
  }

  def isSCMDataEmpty(scmData: SCMTraceabilityData): Boolean = scmData == null ||
    scmData != null &&
      scmData.getAuthor == null &&
      scmData.getCommit == null &&
      scmData.getFileName == null &&
      scmData.getKind == null &&
      scmData.getMessage == null &&
      scmData.getRemote == null &&
      scmData.getDate == null

  def joinPaths(path: Seq[String]): String = {
    path.foldLeft("")((acc, path) => Ids.SEPARATOR.mkString(acc, "", path)).stripMargin('/')
  }

  def whenEnabled[T](isEnabled: CisConfig => Boolean)(callback: => List[T])(implicit generatorConfig: GeneratorConfig): List[T] = {
    if (isEnabled(generatorConfig.cisConfig)) {
      callback
    } else {
      Nil
    }
  }

  def variableToFormVariable(variable: Variable): forms.Variable = {
    val formVariable = new forms.Variable()
    formVariable.setDescription(variable.getDescription)
    formVariable.setKey(variable.getKey)
    formVariable.setLabel(variable.getLabel)
    formVariable.setRequiresValue(variable.getRequiresValue)
    formVariable.setShowOnReleaseStart(variable.getShowOnReleaseStart)
    formVariable.setType(variable.getType.toString)
    formVariable.setValue(variable.getValue)
    formVariable.setValueProvider(variable.getValueProvider)
    if (Type.valueOf(formVariable.getType) == Type.valueOf(classOf[PasswordStringVariable])) {
      val externalValue = variable.asInstanceOf[PasswordStringVariable].getExternalVariableValue
      if (externalValue != null) {
        formVariable.setExternalVariableValue(extVarValueToFormExtVarValue(externalValue))
      }
    }
    if (variable.hasProperty("multiline")) {
      formVariable.setMultiline(variable.getProperty("multiline"))
    }

    formVariable
  }

  def extVarValueToFormExtVarValue(externalVariableValue: ExternalVariableValue): forms.ExternalVariableValue = {
    val formExternalVariableValue = new forms.ExternalVariableValue()
    formExternalVariableValue.setServerType(externalVariableValue.getServer.getType.toString)
    formExternalVariableValue.setServer(externalVariableValue.getServer.getId)
    formExternalVariableValue.setPath(externalVariableValue.getPath)
    formExternalVariableValue.setExternalKey(externalVariableValue.getExternalKey)
    formExternalVariableValue
  }


  // See https://stackoverflow.com/a/1153025
  // Adapted for use of ? and * as wildcards in search
  def stringLike(str: String, expr: String): Boolean = {
    val regex = quotemeta(expr).replace("?", ".?").replace("*", ".*")
    val p = Pattern.compile(regex, Pattern.CASE_INSENSITIVE | Pattern.DOTALL)
    p.matcher(str).matches()
  }

  private def quotemeta(s: String) = {
    if (Strings.isBlank(s)) {
      ""
    } else {
      val sb = new StringBuilder()
      for (i <- 0 until s.length) {
        val c = s.charAt(i)
        if ("[](){}.+$^|#\\".indexOf(c) != -1) {
          sb.append("\\")
        }
        sb.append(c)
      }
      sb.toString()
    }

  }

  // Sort template cis so templates with create release tasks are processed last.
  def sortReleaseCis(ci1: ConfigurationItem, ci2: ConfigurationItem): Boolean = {
    (ci1, ci2) match {
      case (r1: Release, r2: Release) =>
        if (!r1.getAllTasks.asScala.exists(t => t.isInstanceOf[CreateReleaseTask]) && r2.getAllTasks.asScala.exists(t => t.isInstanceOf[CreateReleaseTask])) {
          return true
        }
      case _ =>
    }
    false
  }

  def ciSortOrderValue(x: Any): Int = {
    //assign a sort order for ci types to control processing order
    x match {
      case _: Configuration => 1
      case _: GlobalVariables => 2
      case _: FolderVariables => 3
      case _: Delivery => 4
      case _: RiskProfile => 5
      case _: Release => 6
      case _: Dashboard => 7
      case _: Trigger => 8
      case _: Folder => 9
      case _: BaseConfiguration => 10
      case _: BaseConfigurationItem => 11
      case _ => 12
    }
  }

  def parseAbsolutePath(path: String): FolderPathAndTitle = {
    val parts = splitStringByPathSeparator(path)
    if (parts.length == 1) {
      FolderPathAndTitle(None, unescapeJava(path))
    } else {
      FolderPathAndTitle(Some(joinPaths(parts.slice(0, parts.length - 1))), unescapeJava(parts.last))
    }
  }

  def processCisForYaml(cis: List[ConfigurationItem]): Unit = {
    val escapedPropertyNames = Seq("description", "script", "body", "bulkBody", "expr", "condition", "precondition", "authenticationScript", "failureHandler")
    val ignoreScriptCiTypes = Seq("remoteScript.Windows", "remoteScript.WindowsSmb", "remoteScript.WindowsSsh")

    CiHelper.getNestedCis(cis.asJava).forEach(ci => {
      ci.getType.getDescriptor.getPropertyDescriptors.asScala.foreach {
        case pd if pd.getKind == STRING && pd.getSize == LARGE && pd.isPassword == false && !pd.getName.equals("script") => {
          pd.set(ci, formatForYamlBlockLiteral(Option(pd.get(ci).asInstanceOf[String])).getOrElse(null))
        }
        case pd if escapedPropertyNames.contains(pd.getName) => {
          if (!ci.isInstanceOf[PythonScript] || !(ignoreScriptCiTypes.contains(ci.asInstanceOf[PythonScript].getCustomScriptTask.getPythonScript.getType.toString) && pd.getName.equals("script"))) {
            pd.set(ci, formatForYamlBlockLiteral(Option(pd.get(ci).asInstanceOf[String])).getOrElse(null))
          }
        }
        case _ =>
      }
    })
  }

  def formatForYamlBlockLiteral(value: Option[String]): Option[String] = {
    value match {
      case Some(value) => Some(value.replaceAll("\t", "    ") //replace tabs with spaces
        .replaceAll("\r\n?", "\n") //convert to unix style line endings
        .replaceAll("(?m) +$", "")) //remove white space at the end of all lines
      case _ => None
    }
  }

  /**
   * Determine if permissions should be enforced when processing the specific folder for as-code generate.
   * Permissions should be enforced if user is not admin and does not have FOLDER_GENERATE permission for the folder
   */
  def enforcePermissionsOnGenerate(config: CiGenerateConfig, folderId: String): Boolean = {
    !filterFolderOnGeneratePermissions(config, folderId)
  }

  /**
   * Determine if the folder should be filtered on as-code generate based on current permissions
   */
  def filterFolderOnGeneratePermissions(config: CiGenerateConfig, folderId: String): Boolean = {
    config.isAdminUser || config.folderGeneratePermissions.getOrElse(folderId, false)
  }
}
