package com.xebialabs.xlrelease.ascode.schema

import com.xebialabs.ascode.schema.JsonSchema
import com.xebialabs.ascode.schema.JsonSchema._
import com.xebialabs.ascode.yaml.sugar.Sugarizer
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, PropertyDescriptor, Type}
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.ascode.yaml.sugar.XLRSugar
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.collection.immutable.ListMap
import scala.jdk.CollectionConverters._

object JsonSchemaGenerator {

  val productPrefix: String = "xlr/definitions/"
  val prefix: String = buildPrefix(productPrefix)

  private val configType = Type.valueOf(classOf[Configuration])
  private val configTypes = configType :: DescriptorRegistry.getSubtypes(configType).asScala.toList
  val rootReferences: List[SchemaNode] = buildRefList(configTypes ++ List(classOf[Release], classOf[RiskProfile], classOf[Dashboard]).map(Type.valueOf(_)), productPrefix)

  val dirDesc: SchemaNode = refNode(s"${buildPrefix(productPrefix)}.xlrelease.Folder")

  val sugarNodesTypes: Iterable[Type] = XLRSugar.config.typeToDescriptors.values.filter(_.key.isDefined).map(_.ciType)

  val blacklistedRequiredProperties = List("xlrelease.Phase.color", "xlrelease.CreateReleaseTask.folderId")

}


@Service
class JsonSchemaGenerator @Autowired()(parser: XLRDefinitionParser) {

  import JsonSchemaGenerator._

  private implicit val generatorContext: SchemaGeneratorContext = SchemaGeneratorContext(DescriptorRegistry.getDescriptors.asScala.toList)

  private lazy val sugarNodes = XLRSugar.config.typeToDescriptors.values.filter(_.key.isDefined).map { d =>
    val descr = d.ciType.getDescriptor
    val field = d.key.get
    (s"$xlNamespace.${descr.getType.toString}", getSugaredTypeDefinition(descr, field))
  }.toMap


  private val scriptTaskType = Type.valueOf(classOf[Task])

  private def pythonScriptNodes() = {
    val pythonDescriptors = generatorContext.allTypes.filter(_.getType.instanceOf(Type.valueOf(classOf[PythonScript])))
    pythonDescriptors.map { d =>
      (s"$xlNamespace.${d.getType.toString}.pythonTask",getTypeDefinition(d, scriptTaskType.getDescriptor.getPropertyDescriptors.asScala.toList.filterNot(_.getName.equals("status"))))
    }
  }.toMap

  private lazy val generators =
    parser
      .specs
      .specs
      .flatMap {
        case ((_, kind), generator: SchemaGenerator) => Some(kind -> generator)
        case _ => None
      }

  private def kindNodes() =
    generators
      .map { case (kind, generator) => generator.generateSchema(kind) }
      .toList


  private lazy val xlrSchema: SchemaNode = node
    .id("https://xebialabs.com/ascode.schema.json")
    .`type`("object")
    .schema("https://json-schema.org/draft-07/schema#")
    .description("A description of objects in the XL Products")
    .oneOf(kindNodes())
    .definitions(sugarNodes.map { case (k, v) => (s"$k.sugar", v) } ++ getAllDefinitions ++ pythonScriptNodes())

  lazy val YamlSchema: SchemaNode = JsonSchema.buildSchemaMerge("xlr", xlrSchema)

  private def getAllDefinitions: Map[String, ListMap[String, Any]] =
    getDirectoryDefinition ++
    ListMap(generatorContext.allTypes.filterNot(_.getType.instanceOf(Type.valueOf(classOf[Folder]))).map(desc => (s"$xlNamespace.${desc.getType.toString}", getTypeDefinition(desc))): _*)

  def getTypeDefinition(d: Descriptor, extraDescriptors: List[PropertyDescriptor] = List())(implicit generatorContext: SchemaGeneratorContext): SchemaNode = {
    val allDescriptors = d.getPropertyDescriptors.asScala ++ extraDescriptors
    val pds = allDescriptors.filterNot(_.isHidden).toList.filter { p =>
      if(p.getName.equals("title")) {
        !XLRSugar.config.keysToDescriptors.get("title").exists(_.generateHints.idField.contains("title"))
      } else {
        Sugarizer.writeWithSugar(p.getName, "")(XLRSugar.config, d.getType).isDefined}
      }

    node
      .`type`("object")
      .description(Option(d.getDescription).getOrElse("No description available"))
      .additionalProperties(false)
      .required("type" :: getAllRequiredProperties(pds).distinct: _*)
      .properties(
        getStandardProperties(d.getType) ++
          getPropertyDefinitions(d.getType, pds)
      )
  }

  def getSugaredTypeDefinition(d: Descriptor, key: String)(implicit generatorContext: SchemaGeneratorContext): SchemaNode = {
    val pds = d.getPropertyDescriptors.asScala.filterNot(_.isHidden).toList.filter { p =>
      !p.getName.equals("title") && Sugarizer.writeWithSugar(p.getName, "")(XLRSugar.config, d.getType).isDefined
    }

    node
      .`type`("object")
      .description(Option(d.getDescription).getOrElse("No description available"))
      .additionalProperties(false)
      .required(key :: getAllRequiredProperties(pds): _*)
      .properties(
        ListMap(key -> node.`type`("string").description("The name of the CI")) ++
        getPropertyDefinitions(d.getType, pds)
      )
  }


  private def getAllRequiredProperties(pd: List[PropertyDescriptor]) = pd.filter(pd => pd.isRequired &&
    !pd.getName.equals("title") &&
    !pd.getCategory.equals("output") &&
    !pd.getCategory.equals("internal") &&
    !blacklistedRequiredProperties.exists(_.equals(pd.getFqn)) &&
    !pd.isAsContainment &&
    (pd.getDefaultValue == null || pd.getDefaultValue.equals("")))
    .map(_.getName)

  private def getStandardProperties(t: Type) = ListMap(
    "name" -> node.`type`("string").description("The name of the CI"),
    "type" -> node.`type`("string").description("The type of the CI").enum(t.toString)
  )

  private def getPropertyDefinitions(t: Type, pd: List[PropertyDescriptor]) = {
    ListMap(pd.flatMap { p =>

      if (p.getName.equals("title")) {
        Some(XLRSugar.config.keysToDescriptors.get("title").flatMap(_.generateHints.idField).getOrElse("title"), getPropertyDefinition(p))
      } else {
        Sugarizer.writeWithSugar(p.getName, "")(XLRSugar.config, t).map { case (f, _) =>
          (f, getPropertyDefinition(p))
        }
      }

    }: _*)
  }

  private def getPropertyDefinition(prop: PropertyDescriptor) = node
    .description(prop.getDescription)
    .mergePart(kindToJsonType(prop))
    .mergePart(
      Option(prop.getDefaultValue)
        .map(v => ListMap("default" -> v))
        .getOrElse(ListMap.empty)
    )

  private def kindToJsonType(pd: PropertyDescriptor) = pd.getKind match {
    case BOOLEAN => node.`type`("boolean")
    case INTEGER => node.`type`("integer")
    case ENUM => node.`type`("string").enum(pd.getEnumValues.asScala.toSeq: _*)
    case xs@(LIST_OF_STRING | SET_OF_STRING) => node
      .`type`("array")
      .uniqueItems(xs != LIST_OF_STRING)
      .items(node.`type`("string"))
    case MAP_STRING_STRING => node
      .`type`("object")
      .patternProperties(".*" -> node.`type`("string"))
    case xs@(LIST_OF_CI | SET_OF_CI) =>
      val collectionNode = node.`type`("array")
      val referencedType = pd.getReferencedType
      val subtypes = DescriptorRegistry.getSubtypes(referencedType).asScala.toList
      if (pd.isAsContainment) {
        val allTypes = referencedType :: subtypes
        val sugarRefs = allTypes.filter(t => sugarNodesTypes.toList.contains(t)).map(r => refNode(s"$prefix.${r.toString}.sugar"))

        val pythonTasks = generatorContext.allTypes.filter(_.getType.instanceOf(Type.valueOf(classOf[PythonScript]))).map(d => d.getType.toString)

        val pythonTaskrefs = if(referencedType.instanceOf(Type.valueOf(classOf[Task]))) pythonTasks.map(r => refNode(s"$prefix.$r.pythonTask")) else List()

        collectionNode.items(node.oneOf(buildRefList(allTypes, productPrefix) ++ sugarRefs ++ pythonTaskrefs))
      } else {
        collectionNode
          .items(node.`type`("string"))
          .uniqueItems(xs != LIST_OF_CI)
      }
    case _ => node.`type`("string")
  }

  private def getDirectoryDefinition = {
    val dirRef = refNode(s"$prefix.xlrelease.Folder")

    val sugarrefs = JsonSchemaGenerator.sugarNodesTypes.map(t => refNode(s"$prefix.${t.toString}.sugar"))

    val allDirRefs = dirRef :: rootReferences ++ sugarrefs

    val childrenNode = node
      .`type`("array")
      .description("The children")
      .items(node.oneOf(allDirRefs))

    definitions(
      s"$xlNamespace.xlrelease.Folder" -> node
        .`type`("object")
        .description("Folder")
        .additionalProperties(false)
        .required("type", "name")
        .properties(getStandardProperties(Type.valueOf(classOf[Folder])) + ("name" -> node.`type`("string")) + ("children" -> childrenNode)),

      s"$xlNamespace.xlrelease.Folder.sugar" -> node
        .`type`("object")
        .description("Folder")
        .additionalProperties(false)
        .required("directory")
        .properties(
          "directory" -> node
            .`type`("string")
            .description("Name of the Folder"),
          "children" -> childrenNode
        )
    )
  }

}

