package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.RiskApi
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.risk.domain.riskassessors.RiskAssessor
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class RiskProfileAsCodeService @Autowired()(riskApi: RiskApi) extends Logging {

  private def getProperRiskProfileAssessors(baseAssessors: List[RiskAssessor], riskProfile: RiskProfile): Map[String, String] = {
    val inputAssessors: Map[String, String] = riskProfile.getRiskProfileAssessors.asScala.toMap
    val defaultRiskAssessors = baseAssessors.map { riskAccessor =>
      riskAccessor.getType.toString -> riskAccessor.getScore.toString
    }.toMap

    val hasInvalidAccessors = !inputAssessors.exists {
      case (accessorType, score) => defaultRiskAssessors.contains(accessorType) && isValidScore(score.toInt)
    } && inputAssessors.nonEmpty

    if (hasInvalidAccessors) {
      throw new AsCodeException(s"The risk profile with name [${riskProfile.getTitle}] got some invalid Input Assessors. Make sure that the keys are contained in the default list of Assessors and the values are between 0 and 100.")
    }

    defaultRiskAssessors ++ inputAssessors
  }

  private def isValidScore(value: Int): Boolean = 0 <= value && value <= 100

  def createOrUpdateRiskProfile(riskProfile: RiskProfile): ChangedIds = {
    if (riskProfile.getTitle == null || riskProfile.getTitle.isEmpty) {
      throw new AsCodeException("Risk profile should have title property.")
    }

    val curatedRiskProfileAssessors = getProperRiskProfileAssessors(riskApi.getAllRiskAssessors.asScala.toList, riskProfile)

    riskProfile.setRiskProfileAssessors(curatedRiskProfileAssessors.asJava)

    try {
      val existingRiskProfile = riskApi.getRiskProfileByTitle(riskProfile.getTitle)
      riskProfile.setId(existingRiskProfile.getId)
      val updatedRiskProfile = riskApi.updateRiskProfile(riskProfile)

      CI.ids.withUpdated(updatedRiskProfile.getId)
    } catch {
      case _: NotFoundException =>
        val createdRiskProfile = riskApi.createRiskProfile(riskProfile)
        CI.ids.withCreated(createdRiskProfile.getId)
      case ex: Throwable => throw ex
    }
  }
}
