package com.xebialabs.xlrelease.versioning.ascode.actors

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.scm.data.VersionInfo
import com.xebialabs.xlrelease.versioning.ascode.actors.FolderVersioningActor._
import grizzled.slf4j.Logging
import org.apache.pekko.pattern.ask
import org.apache.pekko.util.Timeout
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.Date
import scala.concurrent.{Await, TimeoutException}

@Component
class FolderVersioningActorService @Autowired()(folderVersioningActorHolder: FolderVersioningActorHolder, xlrConfig: XlrConfig) extends Logging {
  private lazy val actorRef = folderVersioningActorHolder.actorRef()

  def fetchChanges(folderId: String): (Date, Seq[VersionInfo]) = askAndAwait {
    FetchChanges(folderId)
  }

  def createVersion(folderId: String, version: String, description: String): VersionInfo = askAndAwait {
    CreateVersion(folderId, version, description)
  }

  def createOrUpdateSettings(config: FolderVersioningSettings): FolderVersioningSettings = askAndAwait {
    CreateOrUpdateSettings(config)
  }

  def cleanLocalRepo(folderId: String, clusterWide: Boolean): Unit = {
    actorRef ! CleanLocalRepo(folderId, clusterWide)
  }

  def resetLocalRepo(settings: FolderVersioningSettings, clusterWide: Boolean): Unit = {
    actorRef ! ResetLocalRepo(settings, clusterWide)
  }

  def clearSecrets(folderId: String): Unit = {
    actorRef ! ClearSecrets(folderId)
  }

  private def askAndAwait[T](msg: AnyRef): T = {
    implicit val askTimeout: Timeout = xlrConfig.timeouts.releaseActionResponse
    try {
      Await.result(
        actorRef ? msg,
        askTimeout.duration
      ).asInstanceOf[T]
    } catch {
      case e: TimeoutException => throw new RuntimeException(e)
      case e: InterruptedException => throw new RuntimeException(e)
    }
  }
}
