package com.xebialabs.xlrelease.ascode.service.previewhandler
import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.PreviewResponse
import com.xebialabs.ascode.yaml.dto.PreviewResponse.{TaskInfo, TaskPreviewItem}
import com.xebialabs.xlrelease.api.v1.TemplateApi
import com.xebialabs.xlrelease.ascode.service.FolderAsCodeService
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.yaml.model.StartReleaseSpec
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.service.ReleaseService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
@Autowired
class PreviewReleaseHandler(templateApi: TemplateApi,
                            folderAsCodeService: FolderAsCodeService,
                            releaseService: ReleaseService) extends PreviewHandler {
  override def isDefinedAt(config: PreviewConfig): Boolean = config.spec.isInstanceOf[StartReleaseSpec]

  override def apply(config: PreviewConfig): PreviewResponse = {
    val startRelease = config.spec.asInstanceOf[StartReleaseSpec]
    val templateTitle = Ids.getName(startRelease.templatePath)
    val templateFolder = folderAsCodeService.searchParentFolder(startRelease.templatePath, config.metadata)
    Utils.filterCisByFolder(
      releaseService.findTemplatesByTitle(templateFolder.map(_.getId).getOrElse(Ids.SEPARATOR), templateTitle, 0, 2, 1).asScala.toList,
      templateFolder.map(_.getId).getOrElse(Ids.SEPARATOR)
    ) match {
      case template :: Nil =>
        val releaseFolder = folderAsCodeService.searchParentFolder(startRelease.releaseName, config.metadata)
        val releaseFolderPath = Utils.getFolderPathFromCiPath(startRelease.releaseName)
        val requiredVariables = template.getVariables.asScala.filter(variable => variable.getShowOnReleaseStart && variable.getRequiresValue)
          .filterNot(variable => startRelease.variables.contains(variable.getKey))

        if(requiredVariables.nonEmpty) {
          throw new AsCodeException(s"The following variables must be defined in the YAML file to start the release: [${requiredVariables.map(_.getKey).mkString(", ")}].")
        }

        releaseFolderPath.map(folderPath => releaseFolder.getOrElse(throw new AsCodeException(s"The folder [$folderPath] does not exist in the system.")))

        PreviewResponse(Some(TaskInfo(s"Release is will be created from template [${startRelease.templatePath}]", List(TaskPreviewItem(startRelease.releaseName, Nil)))))
      case Nil => throw new AsCodeException(s"No templates with the name [${startRelease.templatePath}] were found.")
      case _ => throw new AsCodeException(s"There is more than one template with the name [${startRelease.templatePath}]. Please introduce an unique name.")
    }
  }
}
