package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.{Utils => CommonUtils}
import com.xebialabs.ascode.yaml.dto.AsCodeResponse
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.model.permission.{GlobalPermissionRelation, PermissionRelation, RolePermissionsRelation}
import com.xebialabs.deployit.core.api.dto.RolePermissions
import com.xebialabs.deployit.core.rest.api.SecurityResource
import com.xebialabs.deployit.engine.api.security.Role
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.permission.PlatformPermissions
import com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY
import com.xebialabs.xlrelease.api.internal.{RolePermissionsResource, RolePrincipalsResource, TaskResource}
import com.xebialabs.xlrelease.api.v1.FolderApi
import com.xebialabs.xlrelease.api.v1.views.TeamMemberView.MemberType
import com.xebialabs.xlrelease.api.v1.views.{TaskAccessView, TeamMemberView, TeamView}
import com.xebialabs.xlrelease.ascode.helper.PermissionAsCodeHelper
import com.xebialabs.xlrelease.ascode.service.GenerateService.GeneratorConfig
import com.xebialabs.xlrelease.ascode.service.PermissionsAsCodeService.HOME_MARKER
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.yaml.model.permission.{TaskAccess, TaskAccessRelation, Team, TeamPermissionRelation}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{getParentId, isNullId}
import com.xebialabs.xlrelease.security.{PermissionChecker, XLReleasePermissions}
import com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS
import com.xebialabs.xlrelease.security.XLReleasePermissions.{EDIT_FOLDER_SECURITY, EDIT_FOLDER_TEAMS}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.stream.Collectors
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Service
class PermissionsAsCodeService @Autowired()(permissionAsCodeHelper: PermissionAsCodeHelper,
                                            folderApi: FolderApi,
                                            securityResource: SecurityResource,
                                            rolePrincipalsResource: RolePrincipalsResource,
                                            rolePermissionsResource: RolePermissionsResource,
                                            taskResource: TaskResource,
                                            folderAsCodeService: FolderAsCodeService,
                                            permissions: PermissionChecker) extends Logging {

  private def createUserMember(name: String, memberType: MemberType): TeamMemberView = {
    val teamMember = new TeamMemberView
    teamMember.setType(memberType)
    teamMember.setName(name)

    teamMember
  }

  private def convertToUserMember(userName: String): TeamMemberView = createUserMember(userName, TeamMemberView.MemberType.PRINCIPAL)

  private def convertToRoleMember(roleName: String): TeamMemberView = createUserMember(roleName, TeamMemberView.MemberType.ROLE)

  private def processTeam(inputTeam: Team, foundTeam: Option[TeamView], folderId: String): TeamView = {
    val team = foundTeam.getOrElse(new TeamView)
    val rolesMembers = inputTeam.roles.map(convertToRoleMember)
    val userMembers = inputTeam.users.map(convertToUserMember)
    val members = rolesMembers ::: userMembers

    if (!isNullId(team.getId) && folderId != getParentId(team.getId)) {
      team.setId(null)
    }

    team.setTeamName(inputTeam.name)
    team.setMembers(members.asJava)
    team.setPermissions(inputTeam.permissions.asJava)
    team
  }

  private def processInheritedSystemTeam(team: TeamView): TeamView = {
    team.setId(null)
    team
  }

  private def handleGlobalPermission(globalPermission: GlobalPermissionRelation): AsCodeResponse = {
    val systemRoles = rolePrincipalsResource.readRolePrincipals

    val globalRolePermissions = securityResource.readRolePermissions(GLOBAL_SECURITY_ALIAS, null, null, null)
    val globalRolesToKeep = globalRolePermissions.asScala.filterNot(rolePermission => globalPermission.global.exists(_.role == rolePermission.getRole.getName))

    val (permissions, ids) = globalPermission.global.foldLeft((List.empty[RolePermissions], PERMISSION.ids)) {
      case ((accPermissions, accIds), rolePermission) =>
        val foundRole = systemRoles.asScala.find(_.getRole.getName == rolePermission.role).getOrElse(
          throw new AsCodeException(s"Role [${rolePermission.role}] doesn't exist. Create the role before trying to use it.")
        )

        val permissions = new RolePermissions
        val role = new Role

        role.setName(rolePermission.role)
        role.setId(foundRole.getRole.getId)
        permissions.setPermissions(rolePermission.permissions.asJava)
        permissions.setRole(role)
        (accPermissions :+ permissions, accIds.withUpdated(foundRole.getRole.getId))
    }

    securityResource.writeRolePermissions(GLOBAL_SECURITY_ALIAS, (permissions ++ globalRolesToKeep).asJava)

    AsCodeResponse.ids(ids)
  }

  private def isNonInheritedTeam(storedFolder: Folder, team: TeamView): Boolean = isNullId(team.getId) || storedFolder.getId == getParentId(team.getId)

  private def isInheritedSystemTeam(storedFolder: Folder, team: TeamView): Boolean = {
    team.isSystemTeam && !isNullId(team.getId) && storedFolder.getId != getParentId(team.getId)
  }

  private def handleFolderPermission(teamPermissionRelation: TeamPermissionRelation, home: Option[String]): AsCodeResponse = {
    val directory = home.map { path =>
      if (teamPermissionRelation.directory == HOME_MARKER) path else s"$path/${teamPermissionRelation.directory}"
    }.getOrElse(teamPermissionRelation.directory)

    val storedFolder = folderAsCodeService.searchFolder(directory)
      .getOrElse(throw new AsCodeException(s"Folder $directory doesn't exists."))

    permissions.checkAny(storedFolder.getId, XLReleasePermissions.APPLY_FOLDER_CHANGES, PlatformPermissions.ADMIN)

    val (teamsToModify, folderTeams) = permissionAsCodeHelper.getTeams(storedFolder.getId)
      .asScala.partition(team => teamPermissionRelation.teams.exists(_.name == team.getTeamName))
    val storedTeams = teamsToModify.map(team => team.getTeamName -> team).toMap

    val teamsFromSpec = teamPermissionRelation.teams.map(team => processTeam(team, storedTeams.get(team.name), storedFolder.getId)) ++
      folderTeams.filter(team => isNonInheritedTeam(storedFolder, team)) ++
      folderTeams.filter(team => isInheritedSystemTeam(storedFolder, team)).map(team => processInheritedSystemTeam(team))
    val updatedTeams = permissionAsCodeHelper.deleteAndUpdateFromTeamView(storedFolder.getId, teamsFromSpec.asJava, folderTeams.asJava)

    val changedCis = CI.ids.withUpdated(storedFolder.getId)
    val changedPermissions = updatedTeams.asScala.foldLeft(PERMISSION.ids) { case (acc, team) =>
      team.getTeamName match {
        case teamName if teamsToModify.exists(_.getTeamName == teamName) => acc.withUpdated(team.getId)
        case teamName if folderTeams.exists(_.getTeamName == teamName) => acc
        case _ => acc.withCreated(team.getId)
      }
    }

    AsCodeResponse.ids(changedCis, changedPermissions)
  }

  private def handleTaskAccessPermission(taskAccessRelation: TaskAccessRelation): AsCodeResponse = {
    val globalRoles = rolePrincipalsResource.readRoleNames
    val storedTaskAccess = taskResource.getTaskAccesses

    val (permissions, changedIds) = taskAccessRelation.taskAccess.foldLeft((List.empty[TaskAccessView], PERMISSION.ids)) {
      case ((accPermissions, accIds), taskAccess) =>
        val foundTaskAccess = storedTaskAccess.asScala.find(_.getTaskType == taskAccess.`type`).getOrElse(
          throw new AsCodeException(s"Task type [${taskAccess.`type`}] doesn't exist.")
        )

        taskAccess.roles.foreach { role =>
          if (!globalRoles.contains(role)) {
            throw new AsCodeException(s"Role [${role}] doesn't exist. Create the role before trying to use it.")
          }
        }

        val taskAccessView = new TaskAccessView(foundTaskAccess.getTaskGroup, taskAccess.`type`,
          foundTaskAccess.getLabel, taskAccess.allowedToAll, taskAccess.roles.asJava)

        (accPermissions :+ taskAccessView, accIds.withUpdated(foundTaskAccess.getTaskType))
    }

    taskResource.updateTaskAccesses(permissions.asJava)

    AsCodeResponse.ids(changedIds)
  }

  private def convertTeamViewToTeam(teamView: TeamView): Team = {
    val (principals, roles) = teamView.getMembers.asScala.partition(_.getType == MemberType.PRINCIPAL)
    Team(teamView.getTeamName, roles.map(_.getName).toList, principals.map(_.getName).toList, teamView.getPermissions.asScala.toList)
  }

  def generatePermissions(foldersToGenerate: List[Folder], generatorConfig: GeneratorConfig):
  (List[GlobalPermissionRelation], List[TaskAccessRelation], List[PermissionRelation]) = {
    if (generatorConfig.cisConfig.generatePermissions) {
      generatorConfig.searchScope match {
        case GlobalSearch =>
          (generateGlobalPermissions(generatorConfig.name).toList, generateTaskAccessPermissions(generatorConfig.name).toList, List.empty)
        case FolderSearch(path, id, includeSubFolders) =>
          (List.empty, List.empty, getFolderPermissions(foldersToGenerate, None, Some(path)))
        case AllSearch =>
          (
            generateGlobalPermissions(generatorConfig.name).toList,
            generateTaskAccessPermissions(generatorConfig.name).toList,
            getFolderPermissions(foldersToGenerate)
          )
        case _ => throw new AsCodeException("Not supported")
      }
    } else {
      (List.empty, List.empty, List.empty)
    }
  }

  private def getFolderPermissions(folders: List[Folder], parent: Option[String] = None, home: Option[String] = None): List[PermissionRelation] = {
    folders
      .filter { folder => permissions.hasPermission(EDIT_FOLDER_SECURITY, folder.getId) || permissions.hasPermission(EDIT_FOLDER_TEAMS, folder.getId) }
      .flatMap { folder =>
        val teams = folderApi.getTeams(folder.getId).asScala.map(convertTeamViewToTeam)
        val innerFolders = folder.getChildren.asInstanceOf[java.util.Set[ConfigurationItem]].asScala
          .filter(_.isInstanceOf[Folder])
          .asInstanceOf[mutable.Set[Folder]]
        val builtPath = buildPath(parent, home, folder)
        val childrenPermissions = getFolderPermissions(innerFolders.toList, Some(builtPath), home)
        TeamPermissionRelation(builtPath, teams.toList) :: childrenPermissions
      }
  }

  private def buildPath(parent: Option[String], home: Option[String], folder: Folder): String = {
    if (parent.isEmpty && home.isDefined) {
      HOME_MARKER
    } else {
      (parent.filterNot(_ == HOME_MARKER).toList :+ CommonUtils.escapePath(folder.getTitle)).mkString(Ids.SEPARATOR)
    }
  }

  private def generateGlobalPermissions(name: Option[String]): Option[GlobalPermissionRelation] = {
    if (permissions.hasGlobalPermission(EDIT_SECURITY)) {
      Option(
        rolePermissionsResource
          .getGlobalRolePermissions
          .getRolePermissions
          .stream()
          .filter { rp =>
            name.forall { n => Utils.stringLike(rp.getRole.getName, n) }
          }
          .map[RolePermissionsRelation] { role =>
            RolePermissionsRelation(role.getRole.getName, role.getPermissions.asScala.toList)
          }
          .collect(Collectors.toList[RolePermissionsRelation])
          .asScala
          .toList
      ).filter(_.nonEmpty).map(GlobalPermissionRelation)
    } else {
      None
    }
  }

  private def generateTaskAccessPermissions(name: Option[String]): Option[TaskAccessRelation] = {
    if (permissions.hasGlobalPermission(EDIT_SECURITY)) {
      Option(taskResource
        .getTaskAccesses
        .stream()
        .filter { ta =>
          name.forall { n => Utils.stringLike(ta.getTaskType, n) }
        }
        .map[TaskAccess] { view =>
          TaskAccess(view.getTaskType, view.isAllowedToAll, view.getRoles.asScala.toList)
        }
        .collect(Collectors.toList[TaskAccess])
        .asScala
        .toList
      ).filter(_.nonEmpty).map(TaskAccessRelation.apply)
    } else {
      None
    }
  }

  def handlePermission(permission: PermissionRelation, home: Option[String] = None): ImportResult = {
    permission match {
      case globalPermissions: GlobalPermissionRelation => asCodeResponseToImportResult(handleGlobalPermission(globalPermissions))
      case taskAccessPermissions: TaskAccessRelation => asCodeResponseToImportResult(handleTaskAccessPermission(taskAccessPermissions))
      case teamPermissions: TeamPermissionRelation => asCodeResponseToImportResult(handleFolderPermission(teamPermissions, home))
    }
  }

  private def asCodeResponseToImportResult(asCode: AsCodeResponse): ImportResult = {
    asCode.changes match {
      case Some(changes) =>
        changes.ids.foldLeft(ImportResult(List())) {
          case (acc, ids) => acc.merge(ImportResult(List(ids)))
        }
      case None => ImportResult(List(PERMISSION.ids))
    }
  }
}

object PermissionsAsCodeService {
  val HOME_MARKER = "."
}
