package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, GenerateService}
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings._
import com.xebialabs.xlrelease.domain.{Release, Trigger}
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.scm.connector.ScmException
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.GENERATE_FOLDER_CONFIGURATION
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.MixedMultiFileStrategy.{MULTIPLE_FILES_PER_CI_TYPE, MULTIPLE_FILES_PER_CI_TYPE_VALUE}
import org.springframework.stereotype.Component

object MixedMultiFileStrategy {
  val MULTIPLE_FILES_PER_CI_TYPE = "multipleFilesPerCIType"
  val MULTIPLE_FILES_PER_CI_TYPE_VALUE = "Multiple files per type"
}

/**
 * Some entity types are grouped within one file, some are split across multiple files.
 *
 * One file per entity (every entity of a below listed type will end up in it's own file):
 * - templates
 * - patterns
 * - triggers
 * - dashboards
 *
 * One file per type (all entities of a type go into one file):
 * - variables
 * - notifications
 * - configurations
 * - permissions
 */
@Component
class MixedMultiFileStrategy(permissions: PermissionChecker,
                             folderAsCodeService: FolderAsCodeService,
                             generateService: GenerateService,
                             definitionParser: XLRDefinitionParser) extends BaseMultiFileStrategy {

  override def name: String = MULTIPLE_FILES_PER_CI_TYPE

  override def value: String = MULTIPLE_FILES_PER_CI_TYPE_VALUE

  def generateMultiFileDefinitions(folderId: String, settings: FolderVersioningSettings): Map[String, List[Definition]] = {
    validateGeneratePermissions(folderId)

    val map = scala.collection.mutable.Map[String, List[Definition]]()
    val ciConfigs = splitAndConvert(settings)
    val scope = FolderSearch(folderAsCodeService.getFolderPath(folderId), folderId)
    val excludeIds = Seq(settings.gitConnection.getId)

    ciConfigs.foreach { ciConfig =>
      try {
        val generatorConfig = GeneratorConfig(None, scope, ciConfig, permissions.hasGlobalPermission(ADMIN), excludeIds)
        val definitions = generateService.generate(generatorConfig)

        ciConfig match {
          case c: CisConfig if isFilePerEntity(c) => map.addAll(extractEntities(definitions, settings, c))
          case c: CisConfig if isFilePerType(c) => map.put(getDefinitionsPath(settings.scmPath, resolveFileName(ciConfig)), definitions)
          case _ => ()
        }
      } catch {
        case e: AsCodeException =>
          e.getMessage match {
            case GenerateService.EMPTY_DEFINITION_ERROR =>
            case _ => throw ScmException(e.getMessage, e.getCause)
          }
      }
    }

    map.put(getDefinitionsPath(settings.scmPath), List(generateService.generateImportsSpecDefinition(map.keys.map(getfilename).toList).get))
    map.toMap
  }

  private def isFilePerType(c: CisConfig): Boolean = {
    c.generateVariables || c.generateConfigurations || c.generateNotificationSettings || c.generatePermissions
  }

  private def isFilePerEntity(c: CisConfig): Boolean = {
    c.generateTemplates || c.generateWorkflows || c.generateDeliveryPatterns || c.generateTriggers || c.generateDashboards
  }

  private def extractEntities(definitions: List[Definition],
                              settings: FolderVersioningSettings,
                              cisConfig: CisConfig): IterableOnce[(String, List[Definition])] = {
    definitions.flatMap { definition =>
      val ciSpec = definition.spec.asInstanceOf[CiSpec]
      ciSpec.cis.map { ci =>
        val fileName = getDefinitionsPath(settings.scmPath, resolveFileName(cisConfig, getTitle(ci)))
        fileName -> List(definition.copy(spec = ciSpec.copy(cis = List(ci))))
      }.toMap
    }
  }

  private def resolveFileName(ciConfig: CisConfig, title: String) = {
    val prefix = ciConfig match {
      case c: CisConfig if c.generateTemplates => TEMPLATES_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateWorkflows => WORKFLOWS_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateDeliveryPatterns => PATTERNS_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateTriggers => TRIGGERS_VERSIONING_YAML_FILENAME_PREFIX
      case c: CisConfig if c.generateDashboards => DASHBOARDS_VERSIONING_YAML_FILENAME_PREFIX
      case _ => ""
    }

    s"$prefix$title.yaml"
  }

  private def resolveFileName(ciConfig: CisConfig) = {
    ciConfig match {
      case c: CisConfig if c.generateConfigurations => CONNECTIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateVariables => VARIABLES_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generatePermissions => PERMISSIONS_VERSIONING_YAML_FILENAME
      case c: CisConfig if c.generateNotificationSettings => NOTIFICATIONS_VERSIONING_YAML_FILENAME
      case _ => ""
    }
  }

  private def getTitle(ci: ConfigurationItem): String = {
    ci match {
      case template: Release => template.getTitle
      case dashboard: Dashboard => dashboard.getTitle
      case trigger: Trigger => trigger.getTitle
      case pattern: Delivery => pattern.getTitle
      case _ => ci.getName
    }
  }

  private def validateGeneratePermissions(folderId: String): Unit = {
    if (!(permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId) || permissions.isCurrentUserAdmin)) {
      throw ScmException(s"Unable to generate configuration: must have Admin or 'Generate Configuration' permission on the folder")
    }
  }

  private def splitAndConvert(config: FolderVersioningSettings): Seq[CisConfig] =
    Seq(CisConfig(generateConfigurations = config.exportConfiguration),
      CisConfig(generateDashboards = config.exportDashboards),
      CisConfig(generateTemplates = config.exportTemplates),
      CisConfig(generateWorkflows = config.exportWorkflows),
      CisConfig(generateTriggers = config.exportTriggers),
      CisConfig(generateVariables = config.exportVariables),
      CisConfig(generateDeliveryPatterns = config.exportPatterns),
      CisConfig(generateNotificationSettings = config.exportNotifications),
      CisConfig(generatePermissions = config.exportSecurity)).filter(!_.isEmpty())

  override def definitionParser(): XLRDefinitionParser = definitionParser

  override def maxNumOfFilesGenerated: Integer = Integer.MAX_VALUE
}
