package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.utils.Utils.filterFolderOnGeneratePermissions
import com.xebialabs.xlrelease.ascode.yaml.model.{EmailNotificationSettingsAsCode, NotificationTriggerSettingsAsCode}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.notifications.api.internal.NotificationsResource
import com.xebialabs.xlrelease.notifications.email.EmailNotificationsService
import com.xebialabs.xlrelease.notifications.views.EmailNotificationSettingsView
import com.xebialabs.xlrelease.notifications.views.converters.EmailNotificationSettingsConverter.toView
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER_NOTIFICATIONS
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import java.util.stream.Collectors
import scala.collection.immutable.ListMap

@Component
class NotificationSettingsAsCodeGenerator @Autowired()(notificationsResource: NotificationsResource, permissions: PermissionChecker, emailNotificationsService: EmailNotificationsService)
  extends GenerateStrategy[EmailNotificationSettingsAsCode] {
  /**
   * This order will be used to sort generate strategies
   */
  override val generateOrder: Double = 200

  override def isDefinedAt(config: CiGenerateConfig): Boolean = {
    config.generateConfig.generateNotificationSettings
  }

  override protected def generateFolder(config: CiGenerateConfig): Map[Option[Folder], List[EmailNotificationSettingsAsCode]] = {
    val emailNotificationSettings = config.folders
      .filter { folder => filterFolderOnGeneratePermissions(config, folder.getId) || permissions.hasPermission(EDIT_FOLDER_NOTIFICATIONS, folder.getId) }
      .map { folder =>
        Some(folder) -> List(getEmailNotificationSettings(Some(folder.getId), config))
      }
    ListMap(emailNotificationSettings: _*)
  }

  override protected def generateGlobal(config: CiGenerateConfig): Map[Option[Folder], List[EmailNotificationSettingsAsCode]] = {
    if (config.isAdminUser) {
      ListMap(None -> List(getEmailNotificationSettings(None, config)))
    } else {
      ListMap.empty
    }
  }

  private def getEmailNotificationSettings(folderId: Option[String], config: CiGenerateConfig): EmailNotificationSettingsAsCode = {
    val emailNotificationSettingsView: EmailNotificationSettingsView = folderId match {
      case Some(id) => {
        if (!config.folderGeneratePermissions.getOrElse(id, false)) {
          permissions.checkEditNotification(id)
        }
        toView(emailNotificationsService.getSettings(id))
      }
      case None => notificationsResource.getEmailNotificationSettings
    }

    val emailNotificationSettings = new EmailNotificationSettingsAsCode
    emailNotificationSettings.setId("emailId")

    emailNotificationSettings.notifications = emailNotificationSettingsView
      .getEmailNotificationSettings
      .stream()
      .map[NotificationTriggerSettingsAsCode] { s =>
        val nts = new NotificationTriggerSettingsAsCode
        nts.setId(s"emailId/${s.getNotificationTriggerName}")
        nts.notification = s.getNotificationTriggerName
        nts.priority = s.getPriority
        nts.subject = Option(s.getTemplateSettings).map(_.getSubject).orNull
        nts.body = Option(s.getTemplateSettings).map(_.getBody).orNull
        nts.bulkSubject = Option(s.getBulkTemplateSettings).map(_.getSubject).orNull
        nts.bulkBody = Option(s.getBulkTemplateSettings).map(_.getBody).orNull
        nts.roles = Option(s.getRecipientSettings).map(s => new util.HashSet[String](s.getRoles)).orNull
        nts.globalRoles = Option(s.getRecipientSettings).map(s => new util.HashSet[String](s.getGlobalRoles)).orNull
        nts.users = Option(s.getRecipientSettings).map(s => new util.HashSet[String](s.getUsers)).orNull
        nts
      }
      .collect(Collectors.toList[NotificationTriggerSettingsAsCode])
    emailNotificationSettings
  }

  override protected def filter(results: Map[Option[Folder], List[EmailNotificationSettingsAsCode]],
                                ciTitle: String): Map[Option[Folder], List[EmailNotificationSettingsAsCode]] = {
    results.map { case (maybeFolder, cis) =>
      val notificationSettings = cis.head
      // update notification trigger list with filtered list
      notificationSettings.notifications = notificationSettings.notifications
        .stream()
        .filter(nts => Utils.stringLike(nts.notification, ciTitle))
        .collect(Collectors.toList[NotificationTriggerSettingsAsCode])

      if (notificationSettings.notifications.isEmpty) {
        maybeFolder -> List.empty
      } else {
        maybeFolder -> cis
      }
    }
  }
}
