package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.ascode.utils.DateUtils._
import com.xebialabs.xlrelease.ascode.utils.{ImportContext, ImportScope, TemplateScope, Utils}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.{CreatedFromAsCode, ReleaseCreatedEvent, ReleaseUpdatedFromAsCodeEvent, TemplateVariablesChangedEvent}
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.status.{PhaseStatus, ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.{CiCloneHelper, FacetRepositoryDispatcher, Ids, ReleaseRepository}
import com.xebialabs.xlrelease.service.{CiIdService, FacetService, ReleaseService}
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper._
import com.xebialabs.xlrelease.versioning.ascode.ValidationMessage
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.Date
import scala.collection.immutable.ListMap
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Service
class TemplateAsCodeService @Autowired()(releaseService: ReleaseService,
                                         dashboardAsCodeService: DashboardAsCodeService,
                                         referenceSolver: ReferenceSolver,
                                         releaseRepository: ReleaseRepository,
                                         ciIdService: CiIdService,
                                         eventBus: XLReleaseEventBus,
                                         facetRepositoryDispatcher: FacetRepositoryDispatcher,
                                         facetService: FacetService
                                        ) extends Logging {

  def process(context: ImportContext, template: Release): ImportResult = {

    logger.debug(s"Processing template: ${template.toString} with metadata ${context.metadata.toString}")

    if (!template.getReleaseTriggers.isEmpty) {
      throw new AsCodeException(s"Templates with release triggers are no longer supported.")
    }

    referenceSolver.resolveReferences(template, context.references, context.scope.getFolderId.orNull)

    find(context, template) match {
      case Some(existing) =>
        logger.debug(s"Updating template: ${existing.toString}")
        update(context, existing, template)
      case None =>
        logger.debug(s"Creating template: ${template.toString}")
        val id = context.templateIds.getOrElse(
          templateAbsolutePath(context.scope, template.getTitle),
          generateId(template, context.scope.getFolderId.getOrElse(ROOT_FOLDER_ID))
        )
        create(context, template, id)
    }
  }

  def findOrPredictTemplateIds(allTemplates: ListMap[ImportScope, List[Release]]): Map[String, String] = {
    allTemplates.toList.foldLeft(Map.empty[String, String]) { case (map, (scope, templates)) =>
      val context = ImportContext(scope, Map.empty, List.empty, None)
      map ++ templates.foldLeft(Map.empty[String, String]) { case (idMap, template) =>
        val id = find(context, template) match {
          case None => generateId(template, scope.getFolderId.getOrElse(ROOT_FOLDER_ID))
          case Some(existing) => existing.getId
        }
        idMap + (templateAbsolutePath(scope, template.getTitle) -> id)
      }
    }
  }

  private def templateAbsolutePath(scope: ImportScope, title: String) = {
    scope.getFolderPath match {
      case None => title
      case Some(path) => Utils.joinPaths(Seq(path, title))
    }
  }

  private def find(context: ImportContext, template: Release): Option[Release] = {
    val matches = releaseService.findTemplatesByTitle(context.scope.getFolderId.getOrElse(Ids.SEPARATOR), template.getTitle, 0, 2, 1).asScala.toList

    if (matches.length > 1) {
      logger.debug(s"There are multiple templates named [${template.getTitle}] ${context.scope.description()}, those are: ${matches.toString}")
      throw new AsCodeException(s"Multiple templates are named [${template.getTitle}] ${context.scope.description()}. Can not determine which one to update.")
    }

    matches.headOption
  }

  private def create(context: ImportContext, template: Release, id: String): ImportResult = {
    populateTemplateData(context, template, id)
    // TODO: add folderCiUid to avoid lookup???
    val messages = validate(context, template)
    val created = releaseRepository.create(template, null)

    processScmData(context.scmData, created)

    ImportResult(
      List(CI.ids.withCreated(created.getId)),
      Seq(
        () => eventBus.publish(ReleaseCreatedEvent(created, CreatedFromAsCode(context.scmData)))
      ),
      Map.empty,
      messages
    )
  }


  private def processFacets(facets: Seq[Facet]): Unit = {
    facets.foreach(facet => {
      facetService.validate(facet)
      facetRepositoryDispatcher.create(facet)
    })
  }

  private def update(context: ImportContext, existing: Release, template: Release): ImportResult = {
    populateTemplateData(context, template, existing.getId)
    template.setCiUid(existing.getCiUid)

    val messages = validate(context, template)

    val updated = releaseRepository.replace(existing, template)

    processScmData(context.scmData, updated)
    // Facets are set to null on releaseRepository.replace call, hence creating them here
    processFacets(updated.getAllTasks.asScala.filterNot(_.getFacets.isEmpty).flatMap(_.getFacets.asScala.toSeq).toSeq)

    // sync trigger vars (even if nothing happened)
    val templateVars = updated.getVariables.asScala
      .withFilter(_.getShowOnReleaseStart)
      .map(v => CiCloneHelper.cloneCi(v)).asJava

    ImportResult(
      List(CI.ids.withUpdated(updated.getId)),
      Seq(
        () => eventBus.publish(ReleaseUpdatedFromAsCodeEvent(updated, context.scmData)),
        () => eventBus.publish(TemplateVariablesChangedEvent(updated.getId, templateVars))
      ),
      Map.empty,
      messages
    )
  }

  private def validate(context: ImportContext, template: Release): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(template, context.getFolderInfo()).toList
      case None => List.empty
    }
  }

  private def generateId[T <: ConfigurationItem](ci: T, parentId: String): String = {
    ciIdService.getUniqueId(ci.getType, parentId)
  }

  private def populateTemplateData(context: ImportContext, template: Release, id: String): Unit = {
    // TODO:
    //  - extension use and processDashboard??
    val home = context.scope.getMetadataHome

    // Process template properties first, most others will need to know the id
    template.setId(id)
    template.setStatus(ReleaseStatus.TEMPLATE)

    if (template.getScheduledStartDate == null) {
      template.setScheduledStartDate(new Date)
    }

    if (template.getDueDate == null) {
      if (template.getPlannedDuration == null) {
        val nextHour = addHoursToDate(template.getScheduledStartDate, 1)
        template.setDueDate(nextHour)
      } else {
        template.setDueDate(addHoursToDate(template.getScheduledStartDate, secondsToHours(template.getPlannedDuration)))
      }
    }

    // Process variables next, user input tasks will need to know the ids when fixing references
    template.getVariables.asScala.foreach(variable => {
      fixUpReleaseVariable(variable, template.getId, ciIdService)
    })

    // Resolve defaultTargetFolderId
    referenceSolver.resolveStringReference(template, context.scope.getFolderId.getOrElse(Ids.SEPARATOR), home)

    template.getPhases.asScala.foreach(phase => {
      populatePhaseData(phase, template, context.scope.getFolderId.getOrElse(Ids.SEPARATOR), home, context)
    })

    val templateScopedContext = context.copy(scope = TemplateScope(template.getId, template.getTitle, context.scope))
    template.getExtensions.forEach {
      case d: Dashboard => dashboardAsCodeService.initializeTemplateDashboard(templateScopedContext, d)
    }
  }

  private def populatePhaseData(phase: Phase, template: Release, folderId: String, home: Option[String], context: ImportContext): Unit = {
    phase.setId(generateId(phase, template.getId))
    phase.setStatus(PhaseStatus.PLANNED)
    phase.setRelease(template)

    phase.getTasks.asScala.foreach(task => {
      populateTaskData(task, phase, template, folderId, home, context)
    })
  }

  private def populateTaskData(task: Task,
                               container: TaskContainer,
                               template: Release,
                               folderId: String,
                               home: Option[String],
                               context: ImportContext): Unit = {
    task.setId(generateId(task, container.getId))
    task.setStatus(TaskStatus.PLANNED)
    task.setContainer(container)

    task.getFacets.asScala.foreach(facet => {
      facet.setTargetId(task.getId)
      referenceSolver.resolveStringReference(facet, folderId, home) // resolve for facets
    })

    referenceSolver.resolveStringReference(task, folderId, home, context.templateIds) // resolve for tasks
    scanAndBuildNewVariables(task.getRelease, task, ciIdService)
    task match {
      case customScriptTask: CustomScriptTask => populateCustomScriptTask(customScriptTask)
      case createReleaseTask: CreateReleaseTask => populateCreateReleaseTask(createReleaseTask)
      case gateTask: GateTask => populateGateTask(gateTask)
      case taskGroup: TaskGroup => populateTaskGroup(taskGroup, template, folderId, home, context)
      case inputTask: UserInputTask => populateUserInputTask(inputTask, template.getVariables.asScala)
      case _ =>
    }
  }

  private def populateUserInputTask(task: UserInputTask, variables: mutable.Buffer[Variable]): Unit = {
    val varMap = variables.map(variable => (variable.getKey, variable)).toMap

    val usedVars = task.getVariables.asScala.filter(taskVar => varMap.contains(taskVar.getKey)).map(taskVar => varMap(taskVar.getKey))
    task.setVariables(usedVars.asJava)
  }

  private def populateCustomScriptTask(task: CustomScriptTask): Unit = {
    val pythonScript = task.getPythonScript
    pythonScript.setId(task.getId + "/" + PythonScript.PYTHON_SCRIPT_ID)
    pythonScript.setCustomScriptTask(task)
  }

  private def populateCreateReleaseTask(task: CreateReleaseTask): Unit = {
    task.getTemplateVariables.asScala.foreach(variable => {
      variable.setId(generateId(variable, task.getId))
    })
  }

  private def populateGateTask(task: GateTask): Unit = {
    task.getConditions.asScala.foreach(condition => condition.setId(ciIdService.getUniqueId(condition.getType, task.getId)))
    task.getDependencies.asScala.foreach(dep =>
      if (!VariableHelper.containsVariables(dep.getTargetId)) {
        throw new AsCodeException("Gate tasks with hardcoded release dependencies cannot be applied, change the dependency to using a variable instead.")
      } else {
        dep.setId(ciIdService.getUniqueId(dep.getType, task.getId))
      }
    )
  }

  private def populateTaskGroup(group: TaskGroup, template: Release, folderId: String, home: Option[String], context: ImportContext): Unit = {
    group.getTasks.asScala.foreach(task => populateTaskData(task, group, template, folderId, home, context))

    group match {
      case parallelGroup: ParallelGroup =>
        if (!parallelGroup.getLinks.isEmpty) {
          val nameToTask = parallelGroup.getTasks.asScala.map(task => (task.getTitle, task)).toMap
          parallelGroup.getLinks.asScala.foreach(link => {
            link.setId(generateId(link, group.getId))
            link.setSource(nameToTask(link.getSource.getTitle))
            link.setTarget(nameToTask(link.getTarget.getTitle))
          })
        }
      case _ =>
    }
  }

  private def processScmData(data: Option[SCMTraceabilityData], template: Release): Unit = {
    data.filterNot(Utils.isSCMDataEmpty).foreach(releaseService.createSCMData(template.getId, _))
  }
}
