package com.xebialabs.xlrelease.ascode.yaml.parser

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.TypeSugar._
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.domain.{BaseArtifact, CustomScriptTask, Task}
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xltype.serialization.{CiReader, CiReference, ConfigurationItemConverter}
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

class XLRConfigurationItemParser extends ConfigurationItemConverter with Logging {
  var foundFiles: Map[String, String] = Map[String, String]()
  private val syntheticProperties: scala.collection.mutable.Map[String, AnyRef] = scala.collection.mutable.Map.empty

  private def isInputCorrect(element: String): Boolean = {
    syntheticProperties.get(element).exists {
      case property: String => VariableHelper.containsOnlyVariable(property)
      case _ => false
    }
  }

  private def processScriptPropertyDescriptors(ci: ConfigurationItem,
                                               descriptors: List[PropertyDescriptor],
                                               shouldProcess: String => Boolean): Unit = {
    descriptors.foreach { descriptor =>
      if (shouldProcess(descriptor.getName)) {
        if (descriptor.getKind == PropertyKind.CI) {
          getReferences.add(new CiReference(
            ci,
            descriptor,
            syntheticProperties(descriptor.getName).asInstanceOf[String]
          ))
        }

        if (descriptor.getKind != PropertyKind.CI) {
          ci.setProperty(descriptor.getName, syntheticProperties(descriptor.getName))
        }

        syntheticProperties.remove(descriptor.getName)
      }
    }
  }

  private def fillSyntheticProperties(ci: ConfigurationItem, descriptor: Descriptor, reader: CiReader, ciWithSyntheticProperties: ConfigurationItem): Unit = {
    while (reader.hasMoreProperties) {
      reader.moveIntoProperty()

      try {
        readProperty(reader, descriptor, ci)
      } catch {
        case _: IllegalStateException =>
          val propertyName = if (Utils.beginsWithCaret(reader.getCurrentPropertyName))
            reader.getCurrentPropertyName.tail
          else
            reader.getCurrentPropertyName

          val propertyValue = Option(ciWithSyntheticProperties.getType.getDescriptor.getPropertyDescriptor(propertyName)).map(
            _.getKind match {
              case PropertyKind.MAP_STRING_STRING => reader.getStringMap
              case PropertyKind.LIST_OF_STRING | PropertyKind.SET_OF_STRING => reader.getStringValues
              case _ => reader.getStringValue
            }
          )

          syntheticProperties += propertyName -> propertyValue.getOrElse("")
      }
    }
  }

  private def solveTaskProperties(task: Task, descriptor: Descriptor, reader: CiReader): Unit = {
    val pythonScript = task.asInstanceOf[CustomScriptTask].getPythonScript

    fillSyntheticProperties(task, descriptor, reader, pythonScript)

    processScriptPropertyDescriptors(pythonScript, pythonScript.getInputProperties.asScala.toList, syntheticProperties.contains)
    processScriptPropertyDescriptors(pythonScript, pythonScript.getOutputProperties.asScala.toList, isInputCorrect)

    // skip default properties of PythonScript which are neither input
    // nor output nor transitional, because they are still exported
    // in yaml when using --defaults xl cli option

    pythonScript.getType.getDescriptor.getPropertyDescriptors.asScala.map(_.getName()).foreach(propertyName => {
      if (syntheticProperties.contains(propertyName)) {
        logger.info(s"Task property [${propertyName}] skipped because it is readonly property with default value")
        syntheticProperties.remove(propertyName)
      }
    })
    if (syntheticProperties.nonEmpty) {
      throw new AsCodeException(s"The following fields does not match the type [${descriptor.getType.toString}]: ${syntheticProperties.keys.mkString(", ")}")
    }
    reader.moveOutOfProperty()
  }

  override def readCi(reader: CiReader): ConfigurationItem = {
    typeOf(reader.getType) match {
      case ciType if Utils.isTypeAPythonScript(ciType) =>
        val task: Task = Task.fromType(ciType)
        val descriptor = task.getType.getDescriptor

        if (descriptor == null) {
          throw new IllegalStateException(s"Encountered unknown CI type [$ciType] for ConfigurationItem [${reader.getId}]")
        }

        task match {
          case baseConfigurationItem: BaseConfigurationItem =>
            baseConfigurationItem.set$token(reader.getToken)
            baseConfigurationItem.set$ciAttributes(reader.getCiAttributes)
        }

        solveTaskProperties(task, descriptor, reader)
        getReadCIs.put(task.getId, task)
        task
      case _ => super.readCi(reader)
    }
  }

  override def readProperty(reader: CiReader, descriptor: Descriptor, configurationItem: ConfigurationItem): Unit = {
    reader.getCurrentPropertyName match {
      case "file" if descriptor.getType.isArtifact =>
        val exportedfilePath = reader.getStringValue
        foundFiles = foundFiles + (exportedfilePath -> exportedfilePath)
        val pd = descriptor.getPropertyDescriptor(BaseArtifact.PORTABLE_FILENAME_PROPERTY_NAME)
        pd.set(configurationItem, exportedfilePath)
      case _ =>
        super.readProperty(reader, descriptor, configurationItem)
    }
  }

}
