package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.model.permission.PermissionsSpec
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.ascode.service.spec.{CiSpecInterpreter, PermissionsSpecInterpreter}
import com.xebialabs.xlrelease.versioning.ascode.{ImportValidator, ValidationMessage}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

@Service
@Autowired
class ImportService(ciImporter: CiSpecInterpreter,
                    permissionsImporter: PermissionsSpecInterpreter) {
  def process(importIncludes: ImportIncludes, definition: Definition, scmData: Option[SCMTraceabilityData], validator: Option[ImportValidator]): ImportResult = {
    definition.spec match {
      case ci: CiSpec => ciImporter.importDefinition(importIncludes, definition, scmData, validator)
      case permissions: PermissionsSpec => permissionsImporter.importDefinition(importIncludes, definition, scmData, validator)
      case _ => throw new AsCodeException(s"${definition.spec.getClass} not supported in import service yet")
    }
  }
}

trait DefinitionImporter {
  def importDefinition(importIncludes: ImportIncludes, definition: Definition, scmData: Option[SCMTraceabilityData], validator: Option[ImportValidator]): ImportResult
}

trait PostCommitAction {
  def run(): Unit
}

object ImportResult {
  def empty: ImportResult = ImportResult(List.empty)
}

// ChangedIds should be a list or map keyed by "EntityKind" -> ids
// Merge should find the correct changedIds to merge or add a new entry to the list/map
case class ImportResult(changedIds: List[ChangedIds],
                        postCommitActions: Seq[PostCommitAction] = Seq.empty,
                        // hack. ChangedIds could be changed to contain type and folder information, or we should use a different class
                        changedIdsPerTypePerFolder: Map[String, Map[Type, ChangedIds]] = Map.empty,
                        validationMessages: List[ValidationMessage] = List.empty) {
  def merge(result: ImportResult): ImportResult = {
    val mergedIds = (changedIds ::: result.changedIds)
      .groupBy(_.kind)
      .view
      .mapValues(_.reduce(_.merge(_)))
      .values
      .toList
      .sortBy(_.kind)

    ImportResult(
      mergedIds,
      postCommitActions ++ result.postCommitActions,
      changedIdsPerTypePerFolder ++ result.changedIdsPerTypePerFolder,
      validationMessages ++ result.validationMessages
    )
  }

  def withPostCommitAction(action: PostCommitAction): ImportResult = this.copy(changedIds, postCommitActions :+ action)

  def withPostCommitActions(actions: Seq[PostCommitAction]): ImportResult = this.copy(changedIds, postCommitActions ++ actions)
}


case class ImportIncludes(templates: Boolean = true,
                          workflows: Boolean = true,
                          triggers: Boolean = true,
                          patterns: Boolean = true,
                          dashboards: Boolean = true,
                          configurations: Boolean = true,
                          variables: Boolean = true,
                          notifications: Boolean = true,
                          permissions: Boolean = true)
