package com.xebialabs.xlrelease.versioning.ascode.upgrader

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.xlrelease.builder.FolderBuilder.newFolder
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.Team.{FOLDER_OWNER_TEAMNAME, RELEASE_ADMIN_TEAMNAME, TEMPLATE_OWNER_TEAMNAME}
import com.xebialabs.xlrelease.domain.configuration.HttpConnection
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.folder.Folder.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.scm.connector.Repository
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.security.sql.snapshots.service.PermissionsSnapshotService
import com.xebialabs.xlrelease.service.{FolderService, SharedConfigurationService, TeamService}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.common.BaseInitializingUpgrade
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.MixedMultiFileStrategy.MULTIPLE_FILES_PER_CI_TYPE
import com.xebialabs.xlrelease.versioning.ascode.scm.{FolderVersionApplyHelper, FolderVersioningService}
import com.xebialabs.xlrelease.versioning.ascode.upgrader.XLRelease233ContentFolderInitializer._
import com.xebialabs.xlrelease.versioning.scheduler.FolderVersioningAutoApplyJobService
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class XLRelease233ContentFolderInitializer(folderService: FolderService,
                                           val folderVersioningService: FolderVersioningService,
                                           folderVersioningAutoApplyJobService: FolderVersioningAutoApplyJobService,
                                           sharedConfigurationService: SharedConfigurationService,
                                           permissionSnapshotService: PermissionsSnapshotService,
                                           teamService: TeamService,
                                           xlrConfig: XlrConfig)
  extends BaseInitializingUpgrade with FolderVersionApplyHelper with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "23.3.0#3")

  override def doUpgrade(): Boolean = {
    logger.info("Creating default content folder")

    if (xlrConfig.features.provision.folders.enabled) {
      val daiFolderToCreate = newFolder.withId(DEFAULT_CONTENT_FOLDER_ID).withTitle(xlrConfig.features.provision.folders.defaultContentFolder.folderTitle).build
      val workflowsToCreate = newFolder.withTitle(WORKFLOWS_FOLDER_NAME).build
      val workflowExecutionsToCreate = newFolder.withTitle(WORKFLOW_EXECUTIONS_FOLDER_NAME).build
      Try {
        val daiFolder = folderService.create(ROOT_FOLDER_ID, daiFolderToCreate, createDefaultTeams = false)
        val folderViewerTeam = getFolderViewerTeam()
        teamService.saveTeamsToPlatformWithoutPublishing(daiFolder.getId, folderViewerTeam.asJava, false)

        val workflowsExecutionFolder = folderService.create(daiFolder.getId, workflowExecutionsToCreate, createDefaultTeams = false)
        val workflowAdminTeam = getWorkflowAdminTeam()
        teamService.saveTeamsToPlatformWithoutPublishing(workflowsExecutionFolder.getId, workflowAdminTeam.asJava, false)

        val workflowsFolder = folderService.create(daiFolder.getId, workflowsToCreate, createDefaultTeams = false)

        // create git configuration on root folder
        val gitConfiguration = createGitConfiguration(daiFolder.getId)
        // create folder versioning settings on workflows folder
        createFolderVersioningSettings(workflowsFolder.getId, gitConfiguration)

        val workflowsFetched = fetchAndApplyLatestVersion(workflowsFolder.getId, workflowsFolder.getTitle)
        scheduleAutoApplyGitVersion(workflowsFolder.getId, workflowsFolder.getTitle)

        if (workflowsFetched) {
          // apply versions for all child folders of workflows
          val parentFolder = folderService.findViewableFoldersById(workflowsFolder.getId, enforcePermission = false)
          parentFolder.getChildren.asScala.foreach { children =>
            fetchAndApplyLatestVersion(children.getId, children.getTitle)
            scheduleAutoApplyGitVersion(children.getId, children.getTitle)
          }
        }

        // Event listeners are not subscribed yet, we need to do snapshot manually
        permissionSnapshotService.makeSnapshot(Option(DEFAULT_CONTENT_FOLDER_ID))
        permissionSnapshotService.makeSnapshot(Option(workflowsFolder.getId))
        permissionSnapshotService.makeSnapshot(Option(workflowsExecutionFolder.getId))
      } match {
        case Failure(exception) => logger.error("Failure while creating and importing content for the default content folder", exception)
        case Success(_) => logger.debug(s"Finished creation of default content folder")
      }
    } else {
      logger.info(s"Skipped default content folder creation as folder provisioning is disabled")
    }
    true
  }

  private def scheduleAutoApplyGitVersion(folderId: String, folderName: String): Unit = {
    Try {
      val folderVersioningSettings = folderVersioningService.getSettings(folderId)
      folderVersioningAutoApplyJobService.handleAutoApplyGitVersion(folderVersioningSettings)
    } match {
      case Failure(ex) => logger.error(s"Unable to configure auto apply of folder version for folder '$folderName'", ex)
      case Success(_) => logger.trace(s"Scheduled/unscheduled auto apply of folder version for folder '$folderName")
    }
  }

  private def createGitConfiguration(parentId: String): Repository = {
    val gitConfiguration = Type.valueOf("git.Repository").getDescriptor.newInstance[Repository]("")
    gitConfiguration.setTitle(xlrConfig.features.provision.folders.defaultContentFolder.repositoryTitle)
    gitConfiguration.setUrl(xlrConfig.features.provision.folders.defaultContentFolder.repositoryUrl)
    gitConfiguration.setAuthenticationMethod(HttpConnection.AuthenticationMethod.None)
    gitConfiguration.setFolderId(parentId)

    val createdGitConfiguration = sharedConfigurationService.create(gitConfiguration)
    createdGitConfiguration.asInstanceOf[Repository]
  }

  private def createFolderVersioningSettings(parentId: String, gitRepository: Repository): FolderVersioningSettings = {
    val versionConfiguration = Type.valueOf(classOf[FolderVersioningSettings]).getDescriptor.newInstance[FolderVersioningSettings]("")
    versionConfiguration.setFolderId(parentId)
    versionConfiguration.branch = xlrConfig.features.provision.folders.defaultContentFolder.scmBranch
    versionConfiguration.gitConnection = gitRepository
    versionConfiguration.scmPath = xlrConfig.features.provision.folders.defaultContentFolder.scmPath
    versionConfiguration.exportConfiguration = false
    versionConfiguration.exportDashboards = false
    versionConfiguration.exportNotifications = false
    versionConfiguration.exportPatterns = false
    versionConfiguration.exportTemplates = false
    versionConfiguration.exportWorkflows = false
    versionConfiguration.exportTriggers = false
    versionConfiguration.exportVariables = false
    versionConfiguration.exportSecurity = true
    versionConfiguration.versioningStyle = MULTIPLE_FILES_PER_CI_TYPE
    versionConfiguration.autoImport = true

    folderVersioningService.createOrUpdateSettings(versionConfiguration, remoteValidations = false)
  }

  private def getFolderViewerTeam(): Seq[Team] = {
    val viewer: Team = newTeam(VIEWER_TEAMNAME)
    viewer.getPermissions.add(VIEW_FOLDER.getPermissionName)
    viewer.addRole(GLOBAL_AUTHENTICATED_USERS_ROLE)

    getSystemTeams() :+ viewer
  }

  private def getWorkflowAdminTeam(): Seq[Team] = {
    val workflowAdmin: Team = newTeam(WORKFLOW_ADMIN_TEAMNAME)
    workflowAdmin.getPermissions.add(VIEW_FOLDER.getPermissionName)
    workflowAdmin.getPermissions.add(START_WORKFLOW_EXECUTION.getPermissionName)
    workflowAdmin.addRole(GLOBAL_AUTHENTICATED_USERS_ROLE)
    getSystemTeams() :+ workflowAdmin
  }

  private def newTeam(teamName: String): Team = {
    val team: Team = Type.valueOf(classOf[Team]).getDescriptor.newInstance(null)
    team.setTeamName(teamName)
    team
  }

  private def getSystemTeams(): Seq[Team] = {
    val folderOwner = newTeam(FOLDER_OWNER_TEAMNAME)
    val templateOwner = newTeam(TEMPLATE_OWNER_TEAMNAME)
    val releaseAdmin = newTeam(RELEASE_ADMIN_TEAMNAME)

    Seq(folderOwner, templateOwner, releaseAdmin)
  }
}

object XLRelease233ContentFolderInitializer {
  val DEFAULT_CONTENT_FOLDER_ID = s"${Folder.ROOT_FOLDER_ID}/FolderDefaultReleaseContent"
  val WORKFLOWS_FOLDER_NAME = "Workflows"
  val WORKFLOW_EXECUTIONS_FOLDER_NAME = "Workflow Executions"
  val WORKFLOW_ADMIN_TEAMNAME = "Workflow Admin"
  val VIEWER_TEAMNAME = "Viewer"
}
