package com.xebialabs.xlrelease.ascode.yaml.writer

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator
import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.{Utils => CommonUtils}
import com.xebialabs.ascode.yaml.model.{CiSpec, Spec}
import com.xebialabs.ascode.yaml.sugar.GenerateStrategy.Reject
import com.xebialabs.ascode.yaml.sugar.Sugarizer._
import com.xebialabs.ascode.yaml.sugar.{Add, Change, SugarConfig}
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.ascode.yaml.writer.support._
import com.xebialabs.ascode.yaml.writer.support.descriptor.CustomPropertyDescriptor
import com.xebialabs.ascode.yaml.writer.{GenerateContext, SpecWriter}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.utils.Utils.{escapeWithCaret, isTypeAPythonScript}
import com.xebialabs.xlrelease.ascode.yaml.sugar.XLRSugar._
import com.xebialabs.xlrelease.domain.ScriptHelper.HIDDEN
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.{CustomScriptTask, PythonScript}
import com.xebialabs.xltype.serialization.{CiReference, CiWriter}

import java.util
import java.util.{List => JavaList}
import scala.jdk.CollectionConverters._

object XLRCiWriter extends SpecWriter {
  private val strategy = XLRGeneratorStrategy
  private val pythonScriptProperty = "pythonScript"
  private val excludeCategories = Set("script", HIDDEN, "internal")

  object XLRGeneratorStrategy extends GeneratorStrategy {
    override def transformCi(ci: ConfigurationItem): ConfigurationItem = {
      ci.getType.getDescriptor.getPropertyDescriptors.asScala.find(_.getName == pythonScriptProperty) match {
        case Some(propertyDescriptor) => propertyDescriptor.get(ci).asInstanceOf[PythonScript]
        case None => ci
      }
    }

    def getReferencedCiId(value: Any, references: JavaList[CiReference]): String = {
      references.asScala.find(_.getCi.getId == value.asInstanceOf[ConfigurationItem].getId).map { ciRef =>
        references.remove(ciRef)
        ciRef.getIds.asScala.head
      }.orNull
    }

    override val propertyFilter: GeneratedPropertyFilter = new DefaultGeneratedPropertyFilter()
    override val childrenResolution: Boolean = false

    override def getId(ci: ConfigurationItem, prop: PropertyDescriptor, suffix: Option[String]): String = {
      val (ciType, title) = (ci.getType.toString, Utils.getCiTitle(ci))

      s"$ciType${stringFormat(title)} ${prop.getName}${stringFormat(suffix)}"
    }

    override val resolveTitle: (ConfigurationItem, SugarConfig) => String = { (ci, sugarConfig) =>
      Option(resolveCiKeyValue(ci, sugarConfig)).map(title =>
        if (ci.isInstanceOf[Folder]) {
          CommonUtils.escapePath(title)
        } else {
          title
        }
      ).orNull
    }
  }

  private def listCiProperties(ci: ConfigurationItem)(implicit writerConfig: WriterConfig): LazyList[ConfigurationItem] = {
    ci #:: ci.getType.getDescriptor.getPropertyDescriptors.asScala.to(LazyList).filter(prop =>
      !prop.isHidden && prop.get(ci) != null && strategy.propertyFilter.shouldGenerate(ci, prop, excludeCategories)
    ).flatMap { prop =>
      prop.getKind match {
        case PropertyKind.CI => listCiProperties(prop.get(ci).asInstanceOf[ConfigurationItem])
        case PropertyKind.LIST_OF_CI | PropertyKind.SET_OF_CI =>
          prop.get(ci).asInstanceOf[util.Collection[ConfigurationItem]].asScala.toList.flatMap(listCiProperties)
        case _ => LazyList.empty
      }
    }
  }


  override def validate(spec: Spec)(implicit writerConfig: WriterConfig): Unit = {
    spec.asInstanceOf[CiSpec].cis.to(LazyList).flatMap(listCiProperties).foreach { ci =>
      if (ci != null) {
        ci.getType.getClosestTypeSugar.foreach(descriptor =>
          descriptor.generateHints.strategy match {
            case Reject =>
              throw new AsCodeException(s"Class type [${ci.getClass.getName}] is not supported for generating yet. [$ci]")
            case _ =>
          }
        )
      }
    }
  }

  override def write(spec: Spec, yamlGenerator: YAMLGenerator)(implicit mapper: ObjectMapper, writerConfig: WriterConfig): GenerateContext = {
    val ciSpec = spec.asInstanceOf[CiSpec]
    val writer = new XlrJacksonCiWriter(yamlGenerator)
    val converter = new XlrConfigurationItemGenerator(strategy, excludeCategories)
    converter.getReferences.asScala.addAll(ciSpec.references)
    converter.writeCis(ciSpec.cis.asJava, writer, Int.MaxValue)
    GenerateContext(Nil, converter.secrets)
  }
}

class XlrConfigurationItemGenerator(strategy: GeneratorStrategy = new DefaultGeneratorStrategy(),
                                    excludeCategories: Set[String] = Set())
                                   (implicit sugarConfig: SugarConfig, writerConfig: WriterConfig)
  extends ConfigurationItemGenerator(strategy, excludeCategories)(sugarConfig, writerConfig) {

  lazy val taskProperties: Iterable[PropertyDescriptor] = Type.valueOf(classOf[CustomScriptTask]).getDescriptor.getPropertyDescriptors.asScala

  override def writeProperties(ci: ConfigurationItem, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    if (isTypeAPythonScript(ci)) {
      ci.getType.getDescriptor.getPropertyDescriptors.asScala
        .filter(!_.isHidden)
        .foreach(property => writeProperty(ci, escapePropertyIfNecessary(property), writer, ciRefsFromLevel))
      addSugar(ci, writer, ciRefsFromLevel)
    } else {
      super.writeProperties(ci, writer, ciRefsFromLevel)
    }
  }

  protected def escapePropertyIfNecessary(propertyDescriptor: PropertyDescriptor): PropertyDescriptor = {
    taskProperties.find(pd => pd.getName == propertyDescriptor.getName).fold(propertyDescriptor) { _ =>
      val typeSource = propertyDescriptor.getTypeSource
      CustomPropertyDescriptor(typeSource, delegate = Some(propertyDescriptor), name = Some(escapeWithCaret(propertyDescriptor.getName)))
    }
  }

  // c/p from base class to not change platform for minor release, refactor later
  protected def addSugar(ci: ConfigurationItem, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    ci
      .getType
      .getClosestTypeSugar(sugarConfig)
      .foreach(_.sugarActions.foreach {
        case Add(fieldName, value) => this.writeProperty(
          ci,
          CustomPropertyDescriptor(ci.getType.getTypeSource, name = Some(fieldName), value = Some(value), kind = Some(PropertyKind.STRING)),
          writer,
          ciRefsFromLevel
        )
        case Change(originalFieldName, replaceFieldName) =>
          Option(ci.getType.getDescriptor.getPropertyDescriptor(originalFieldName)).foreach(property =>
            this.writeProperty(
              ci,
              CustomPropertyDescriptor(property.getTypeSource, delegate = Some(property), name = Some(replaceFieldName)),
              writer,
              ciRefsFromLevel
            )
          )
        case _ =>
      })
  }

}
