package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.deployit.security.PermissionDeniedException
import com.xebialabs.xlrelease.api.v1.FolderApi
import com.xebialabs.xlrelease.ascode.service.GenerateService.GeneratorConfig
import com.xebialabs.xlrelease.ascode.service.{AllSearch, ExactSearch, FolderAsCodeService, FolderSearch}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.folder.Folder.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.repository.Ids.SEPARATOR
import com.xebialabs.xlrelease.repository.{Ids, Page}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{GENERATE_FOLDER_CONFIGURATION, VIEW_FOLDER}
import com.xebialabs.xlrelease.service.FolderService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.Optional
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
class FolderGenerator @Autowired()(folderAsCodeService: FolderAsCodeService,
                                   folderApi: FolderApi,
                                   folderService: FolderService,
                                   permissions: PermissionChecker) {
  val TOTAL_RESULT_PER_PAGE = 100L

  def getFolderCis(generatorConfig: GeneratorConfig): List[Folder] = {
    generatorConfig.searchScope match {
      case ExactSearch(ids) => generateFolderByIds(ids)
      case FolderSearch(path, id, includeSubFolders) => generateFolder(path, id, includeSubFolders)
      case AllSearch if generatorConfig.cisConfig.hasFolderTypes() =>
        CiGenerator.paginate { page =>
          val folders = folderService.listViewableFolders(ROOT_FOLDER_ID, Page.parse(Optional.ofNullable(page), Optional.ofNullable(TOTAL_RESULT_PER_PAGE), Optional.ofNullable(Int.MaxValue)), true, false).asScala.toList
          val viewableFolders = filterViewable(folders)
          viewableFolders -> (viewableFolders.size < TOTAL_RESULT_PER_PAGE)
        }
      case _ => List.empty
    }
  }

  private def getFolderHierarchy(id: String): Option[Folder] = {
    if (Ids.ROOT_FOLDER_ID.equals(id)) {
      None
    }
    else {
      var folderId = id
      var root: Folder = null
      val enforcePermissions: Boolean = enforceFolderPermissions(id)
      // build folders from leaf to root
      while (!Ids.ROOT_FOLDER_ID.equals(folderId)) {
        val leaf = folderService.findViewableFoldersById(folderId, 0, enforcePermissions)
        folderId = folderId.substring(0, folderId.lastIndexOf(SEPARATOR))
        if (root != null) {
          leaf.setChildren(Set(root).asJava)
        }
        root = leaf
      }
      Some(root)
    }
  }

  private def generateFolderByIds(ids: List[String]): List[Folder] = {
    val folders: Set[String] = ids.map { id => id.substring(0, id.lastIndexOf(SEPARATOR)) }.toSet
    if (folders.size > 1) {
      // There is no reason why this can't be supported but at the moment it is not needed
      // and supporting it is more trouble than it is worth until there is a use case for it
      throw new AsCodeException("Cannot generate for specific ids in different folders")
    }
    folders.flatMap {
      getFolderHierarchy
    }.toList
  }

  private def generateFolder(path: String, id: String, includeSubFolders: Boolean): List[Folder] = {
    val depth: Integer = if (includeSubFolders) Int.MaxValue else 0
    Try(Option(id).map(_ => Some(folderService.findViewableFoldersById(id, Optional.ofNullable(depth).orElse(Page.DEFAULT_DEPTH), enforceFolderPermissions(id))))
      .getOrElse(folderAsCodeService.searchFolder(path, includeSubFolders))) match {
      case Success(Some(foundFolder)) => filterViewable(foundFolder :: Nil)
      case Failure(e: PermissionDeniedException) => throw e
      case _ => Nil
    }
  }

  private def filterViewable(items: List[Folder]): List[Folder] = {
    items
      .filter(folder => permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folder.getId) || permissions.hasPermission(VIEW_FOLDER, folder.getId))
      .map { folder =>
        folder.setChildren(filterViewable(folder.getChildren.asScala.toList).toSet.asJava)
        folder
      }
  }

  private def enforceFolderPermissions(folderId: String): Boolean = {
    if (permissions.isCurrentUserAdmin || permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId)) false else true
  }
}
