package com.xebialabs.xlrelease.ascode.service.validator

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.xlrelease.ascode.utils.DateUtils
import com.xebialabs.xlrelease.ascode.yaml.model.{Environment, EnvironmentsSpec, Label}
import org.springframework.stereotype.Component

import java.util.Date

@Component
class EnvironmentsSpecValidator extends SpecValidator {
  override def isDefinedAt(definition: Definition): Boolean = definition.spec.isInstanceOf[EnvironmentsSpec]

  private def checkDuplicates[T](list: List[T], kind: String, identityExtractor: T => String): Unit = {
    list
      .groupBy(identityExtractor)
      .find { case (_, elements) => elements.length > 1 }
      .foreach { case (key, _) => throw new AsCodeException(s"$kind '$key' is not unique") }
  }

  override def apply(definition: Definition): Unit = {
    val spec = definition.spec.asInstanceOf[EnvironmentsSpec]
    checkDuplicates[String](spec.stages, "Stage", stage => stage)
    checkDuplicates[Environment](spec.environments, "Environment", _.name)
    checkDuplicates[Label](spec.labels, "Label", _.name)
    val now = new Date()
    spec.reservations.foreach { reservation =>
      val startDate = DateUtils.fromString(reservation.startDate)
      if (startDate.before(now)) {
        throw new AsCodeException(
          s"Cannot create reservation for environment ${reservation.environment} starting before current date [${reservation.startDate}]"
        )
      }
    }
  }
}
