package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.ascode.yaml.model.ImportsSpec
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings.{ALL_FILENAMES, FOLDER_VERSIONING_YAML_FILENAME, getDefinitionsPath, getfilename}
import com.xebialabs.xlrelease.scm.connector.{BinaryFile, ScmBlobs, ScmException}
import com.xebialabs.xlrelease.versioning.ascode.scm.FolderVersioningService.wrapExceptions
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnectorInitializer
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.VersioningStyleResolver
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.nio.charset.StandardCharsets
import scala.util.{Failure, Success, Try, Using}

@Component
class FolderVersioningPreviewService @Autowired()(connectorInitializer: AsCodeJGitConnectorInitializer,
                                                  folderVersioningConfigService: FolderVersioningConfigService,
                                                  versioningStyleResolver: VersioningStyleResolver,
                                                  definitionParser: XLRDefinitionParser) {

  def generatePreview(folderId: String, version: Option[String], fileName: String = FOLDER_VERSIONING_YAML_FILENAME): String = wrapExceptions {
    val config = getSettings(folderId)
    val finalFileName = if (fileName.isEmpty) FOLDER_VERSIONING_YAML_FILENAME else fileName
    val filesToCheckout = getFilesForCheckout(finalFileName, config)
    val blobs: BinaryFile = version match {
      case Some(ver) =>
        generateVersionPreview(finalFileName, config, filesToCheckout, ver)
      case None =>
        generateUnversionedPreview(folderId, finalFileName, config)
    }
    new String(blobs.getContent(), StandardCharsets.UTF_8)
  }

  private def generateVersionPreview(fileName: String, config: FolderVersioningSettings, filesToCheckout: Array[String], ver: String) = {
    val filenameWithPath = getDefinitionsPath(config, fileName)
    Using.resource(connectorInitializer.init(config)) { scmConnector =>
      Try(scmConnector.checkout(filesToCheckout, ver, reset = false)) match {
        case Success(checkedOut) =>
          if (!fileName.equals(FOLDER_VERSIONING_YAML_FILENAME)) {
            verifyRequestedFile(checkedOut, fileName, config)
          }
          getRequestedFile(checkedOut, filenameWithPath)
        case Failure(ex: ScmException) if ex.getMessage.contains("No definition file found for tag") => throw new NotFoundException(ex.getMessage)
        case Failure(ex) => throw ex
      }
    }
  }

  private def generateUnversionedPreview(folderId: String, fileName: String, config: FolderVersioningSettings) = {
    if (!ALL_FILENAMES.contains(fileName)) {
      throw new NotFoundException("Invalid file requested")
    }

    val filePath = getDefinitionsPath(config, fileName)
    val allFiles = versioningStyleResolver.resolve(config.versioningStyle).generateFolder(folderId, config, filePath).blob.files
    allFiles.filter(binaryFile => getfilename(binaryFile.fileName).equals(fileName)).head
  }

  private def getFilesForCheckout(fileName: String, config: FolderVersioningSettings): Array[String] = {
    if (fileName.equals(FOLDER_VERSIONING_YAML_FILENAME)) {
      Array(getDefinitionsPath(config, fileName))
    } else {
      Array(getDefinitionsPath(config, fileName), getDefinitionsPath(config, FOLDER_VERSIONING_YAML_FILENAME))
    }
  }

  def verifyRequestedFile(checkedOut: ScmBlobs, fileName: String, config: FolderVersioningSettings): Unit = {
    val releasefileName = getDefinitionsPath(config)
    val releasefile = checkedOut.files.find(file => file.fileName.equals(releasefileName)).get
    val releasefileYaml = new String(releasefile.getContent(), StandardCharsets.UTF_8)
    val specs = YamlUtils.preprocessYaml(releasefileYaml)
    val releasefileDefinition = definitionParser.parse(specs.head)

    val imports = releasefileDefinition.spec.asInstanceOf[ImportsSpec].imports
    if (!imports.contains(fileName)) {
      throw ScmException("The requested file is not amongst the imports.")
    }
  }

  def getRequestedFile(checkedOut: ScmBlobs, fileName: String): BinaryFile = checkedOut.files.find(_.fileName.equals(fileName)).get

  def getSettings(folderId: String): FolderVersioningSettings = wrapExceptions {
    folderVersioningConfigService.findSettings(folderId).getOrElse(throw ScmException("No version control settings defined for folder"))
  }

}
