package com.xebialabs.xlrelease.ascode.yaml.writer

import com.xebialabs.ascode.yaml.Specs
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.ascode.yaml.writer._
import com.xebialabs.deployit.{ReleaseInfo, ServerConfiguration}
import com.xebialabs.xlrelease.ascode.utils.StaticVariables._

import java.io.{ByteArrayOutputStream, OutputStream}
import java.nio.charset.StandardCharsets
import java.util.Date

object XLRDefinitionWriter {
  private val product = "xl-release"

  private def buildSpecs[T](writers: List[(String, SpecWriter)]): Specs[SpecWriter] = {
    Specs(writers.map { case (kind, writer) => (product, kind) -> writer }.toMap)
  }

  val specs = buildSpecs(List(
    XLR_TEMPLATE_KIND -> XLRCiWriter,
    XLR_PERMISSIONS_KIND -> PermissionWriter,
    XLR_ROLES_KIND -> RolesWriter,
    XLR_USERS_KIND -> XLRUsersWriter,
    XLR_ENVIRONMENTS_KIND -> EnvironmentsWriter,
    XLR_APPLICATIONS_KIND -> ApplicationsWriter,
    XLR_IMPORT_KIND -> ImportsSpecWriter
  ))

  def apply(): XLRDefinitionWriter = {
    new XLRDefinitionWriter()
  }
}

class XLRDefinitionWriter(val specs: Specs[SpecWriter] = XLRDefinitionWriter.specs) extends DefinitionWriter(specs) {

  private def header: String = {
    val releaseInfo = new ReleaseInfo("com/xebialabs/deployit/release.properties")

    s"""|# Exported from:        ${Option(ServerConfiguration.getInstance()).map(_.getServerUrl).getOrElse("Unknown instance")}
        |# Release version:      ${releaseInfo.getVersion}
        |# Date created:         ${new Date()}
        |
        |""".stripMargin
  }

  def writeYaml(out: OutputStream, config: WriterConfig, includeHeader: Boolean = true): List[GenerateContext] = {
    if (includeHeader) {
      out.write(header.getBytes(StandardCharsets.UTF_8))
    }
    config.definitions.map {
      writeDefinitionToStream(out, _)(config)
    }
  }

  def writeZip(out: OutputStream, config: WriterConfig): Unit = {
    write(out, config)
  }

  def writeYamlAsByteArray(config: WriterConfig, includeHeader: Boolean = true): Array[Byte] = {
    val out = new ByteArrayOutputStream()
    writeYaml(out, config, includeHeader)
    out.toByteArray
  }

  def writeYamlAsString(config: WriterConfig, includeHeader: Boolean = true): String = {
    new String(writeYamlAsByteArray(config, includeHeader))
  }
}
