package com.xebialabs.xlrelease.ascode.service.validator

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.utils.{Utils => CommonUtils}
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.domain.folder.Folder
import org.springframework.stereotype.Component

import java.util.{Set => JavaSet}
import scala.jdk.CollectionConverters._

@Component
class CiSpecValidator extends SpecValidator {
  override def isDefinedAt(definition: Definition): Boolean = definition.spec.isInstanceOf[CiSpec]

  private case class DuplicationHelper(id: String, ci: ConfigurationItem)

  private def checkCis(cis: List[ConfigurationItem], parent: String = ""): List[DuplicationHelper] = {
    val (folderCis, otherCis) = cis.partition(_.isInstanceOf[Folder])

    val folders = checkFolders(folderCis.asInstanceOf[List[Folder]], parent)

    folders ::: otherCis.map { ci =>
      val title = Utils.getCiTitle(ci).getOrElse("")
      val id = s"${ci.getType.toString}-${Utils.buildFolderPath(parent, title)}"

      DuplicationHelper(id, ci)
    }
  }

  private def checkFolders(folderCis: List[Folder], parent: String): List[DuplicationHelper] = {
    folderCis.flatMap {
      folder =>
        val folderId = s"Folder-$parent-${folder.getTitle}"
        val (folderChildren, otherChildren) = folder.getChildren.asInstanceOf[JavaSet[ConfigurationItem]].asScala.toList.partition(_.isInstanceOf[Folder])
        val folderPath = Utils.buildFolderPath(parent, folder.getTitle)

        val folderChildrenList = checkFolders(folderChildren.asInstanceOf[List[Folder]], folderPath)
        val otherChildrenList = checkCis(otherChildren, folderPath)

        DuplicationHelper(folderId, folder) :: otherChildrenList ::: folderChildrenList
    }
  }

  private def checkDupe(specCi: List[ConfigurationItem]): Unit = {
    val processedCis = checkCis(specCi)
    val cisList = processedCis.map(_.id)

    val duplicatedCis = CommonUtils.getDuplicatedObjects(cisList)
    if (duplicatedCis.nonEmpty) {
      val filteredCis = processedCis.filter(processedCi => duplicatedCis.contains(processedCi.id)).map(x => x.id -> x.ci).toMap
      val errorMessage = filteredCis.map {
        case (id, ci) =>
          val splittedCi = id.split("-").tail
          val ciType = splittedCi.head
          val title = splittedCi.tail.mkString
          s"""The path [$ciType] with title [$title] is not unique. All ${ci.getType.toString} titles in the template must be unique."""
      }.mkString("\n")
      throw new AsCodeException(s"$errorMessage")
    }
  }

  override def apply(definition: Definition): Unit = {
    val ciSpec = definition.spec.asInstanceOf[CiSpec]
    checkDupe(ciSpec.cis)
  }
}
