package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.utils.TypeSugar.typeOf
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.{ChangedIds, EntityKinds}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.forms.DeliveryPatternFilters
import com.xebialabs.xlrelease.ascode.yaml.model.EmailNotificationSettingsAsCode
import com.xebialabs.xlrelease.delivery.events.DeliveryDeletedEvent
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.service.DeliveryPatternService
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.variables.FolderVariables
import com.xebialabs.xlrelease.domain.{Configuration, Release, Trigger}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.notifications.configuration.EmailNotificationSettings
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.plugins.dashboard.events.DashboardDeletedEvent
import com.xebialabs.xlrelease.plugins.dashboard.repository.SqlDashboardRepository
import com.xebialabs.xlrelease.plugins.dashboard.service.DashboardService
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.{ReleaseConfigurationReferencePersistence, TriggerConfigurationReferencePersistence}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptionsBuilder
import com.xebialabs.xlrelease.service.TeamService
import com.xebialabs.xlrelease.triggers.service.TriggerService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

@Service
class OrphanDeletionService @Autowired()(eventBus: XLReleaseEventBus,
                                         folderPersistence: FolderPersistence,
                                         releaseRepository: ReleaseRepository,
                                         configurationRepository: ConfigurationRepository,
                                         triggerRepository: TriggerRepository,
                                         deliveryPatternService: DeliveryPatternService,
                                         deliveryRepository: DeliveryRepository,
                                         dashboardService: DashboardService,
                                         sqlDashboardRepository: SqlDashboardRepository,
                                         folderVariableRepository: FolderVariableRepository,
                                         triggerService: TriggerService,
                                         teamService: TeamService,
                                         teamRepository: TeamRepository,
                                         releaseConfigurationReferencePersistence: ReleaseConfigurationReferencePersistence,
                                         triggerConfigurationReferencePersistence: TriggerConfigurationReferencePersistence
                                        ) extends Logging {
  /**
   * Anything that is in the YAML will never be orphaned.
   * Anything that isn't in the YAML might or might not be orphaned.
   */
  def processCis(includes: ImportIncludes,
                 topLevelFolder: String,
                 changedCisPerTypePerFolder: Map[String, Map[Type, ChangedIds]],
                 gitRepoId: Option[String]): ImportResult = {
    val changedFolderIds = changedCisPerTypePerFolder.keySet
    // all folders that might be affected by this operation
    // findDescentandsById finds itself as well
    val allRelevantFolders = folderPersistence.findDescendantsById(topLevelFolder).map(_.folderId.absolute).toSet
    // we don't delete orphaned folders, just their contents (because of running releases)
    val foldersToEmpty = allRelevantFolders.diff(changedFolderIds)

    processCiOrphans(
      includes,
      changedCisPerTypePerFolder.concat(foldersToEmpty.map(_ -> Map.empty)),
      gitRepoId
    )

  }

  def processPermissions(includes: ImportIncludes, topLevelFolder: String, changedIds: Iterable[ChangedIds]): ImportResult = {
    // there are no orphan teams when teams are updated thru as-code
    // the new team configuration will reflect the one described in YAML
    // a problem appears when there is no PermissionSpec in YAML, or it's empty somehow
    // then we should reset folder team configuration to initial state
    val deletedIds = ListBuffer.empty[String]
    val postCommitActions = ListBuffer.empty[PostCommitAction]

    if (includes.permissions) {
      // team IDs should contain folder IDs in them
      val changedFolderIds = changedIds.view
        .filter(_.kind == EntityKinds.PERMISSION)
        .flatMap(chids => chids.updated ::: chids.created)
        .map(Ids.findFolderId)
        .toSet
      // all folders that might be affected by this operation
      // findDescentandsById finds itself as well
      val allRelevantFolders = folderPersistence.findDescendantsById(topLevelFolder).map(_.folderId.absolute).toSet
      // these folders will have their teams deleted, subfolders with deleted teams should inherit from parent
      val foldersToEmpty = allRelevantFolders.diff(changedFolderIds)
      foldersToEmpty.foreach { fid =>
        val deletedTeams = if (Ids.getParentId(fid) == Ids.ROOT_FOLDER_ID) {
          // DO NOT delete system teams from a top-level folder, it would break XLR (it has no parent folder to inherit from)
          // we can't tell who te original member of system teams was when the folder was created
          // so we can't perform a correct "reset" of system teams to original members
          // therefore we just won't alter system teams at all
          val (systemTeams, teamsToDelete) = teamService.getEffectiveTeams(fid).asScala.partition(_.isSystemTeam)
          // this will make only system teams remain in the folder
          teamRepository.saveTeamsToPlatform(fid, systemTeams.asJava)
          teamsToDelete
        } else {
          val teamsToDelete = teamService.getEffectiveTeams(fid)
          // this will reset the folder to inherit from parent
          teamRepository.deleteTeamsFromPlatform(fid)
          teamsToDelete.asScala
        }
        // copying code from TeamService#deleteTeamsFromPlatform
        deletedTeams.foreach { team =>
          logger.info(s"Removing orphaned team: ${team.getName}, with id ${team.getId}, from folder ${fid}")
          postCommitActions.addOne(() => eventBus.publish(TeamDeletedEvent(fid, team)))
        }
        if (deletedTeams.nonEmpty) {
          postCommitActions.addOne(() => eventBus.publish(TeamsUpdatedEvent(fid)))
        }
        deletedIds.addAll(deletedTeams.map(_.getId))
      }
    }

    ImportResult(
      List(ChangedIds(EntityKinds.PERMISSION, deleted = deletedIds.toList)),
      postCommitActions.toList
    )
  }

  private def processCiOrphans(includes: ImportIncludes, cisToRetainPerFolder: Map[String, Map[Type, ChangedIds]], gitRepoId: Option[String]): ImportResult = {
    var deletionResult = ImportResult.empty
    cisToRetainPerFolder.foreachEntry { (folderId, cisPerType) =>
      if (includes.triggers) {
        val triggersToRetain = getIdsToRetain(typeOf[Trigger], cisPerType)
        deletionResult = deletionResult.merge(handleTriggers(folderId, triggersToRetain))
      }
      if (includes.templates) {
        val templatesToRetain = getIdsToRetain(typeOf[Release], cisPerType)
        deletionResult = deletionResult.merge(handleTemplates(folderId, templatesToRetain))
      }
      if (includes.dashboards) {
        val dashboardsToRetain = getIdsToRetain(typeOf[Dashboard], cisPerType)
        deletionResult = deletionResult.merge(handleDashboards(folderId, dashboardsToRetain))
      }
      if (includes.patterns) {
        val patternsToRetain = getIdsToRetain(typeOf[Delivery], cisPerType)
        deletionResult = deletionResult.merge(handlePatterns(folderId, patternsToRetain))
      }
      if (includes.variables) {
        val folderVarsToRetain = getIdsToRetain(typeOf[FolderVariables], cisPerType)
        deletionResult = deletionResult.merge(handleFolderVariables(folderId, folderVarsToRetain))
      }
      if (includes.configurations) {
        val configsToRetain = getIdsToRetain(typeOf[Configuration], cisPerType) ++ gitRepoId
        deletionResult = deletionResult.merge(handleConfigurations(folderId, configsToRetain))
      }
      if (includes.notifications) {
        val notificationSettingsToRetain = getIdsToRetain(typeOf[EmailNotificationSettingsAsCode], cisPerType)
        deletionResult = deletionResult.merge(handleNotifications(folderId, notificationSettingsToRetain))
      }
      // other ci stuff: is global so not needed for now
    }

    deletionResult
  }

  private def getIdsToRetain(tpe: Type, cisPerType: Map[Type, ChangedIds]) = {
    val changedIds = cisPerType.getOrElse(tpe, CI.ids)
    (changedIds.updated ::: changedIds.created).toSet
  }

  private def handleTemplates(folderId: String, idsToRetain: Set[String]): ImportResult = {
    val dbIds = releaseRepository.findIdsByStatus(Seq(ReleaseStatus.TEMPLATE): _*).filter { id =>
      val fid = Ids.findFolderId(id)
      fid == folderId
    }.toSet
    // skip templates that are still referenced by triggers
    val (usedTemplates, templatesToDelete) = dbIds.diff(idsToRetain).partition { id =>
      triggerRepository.numberOfTemplateTriggers(id) > 0
    }
    usedTemplates.foreach { id =>
      val templateTitle = releaseRepository.getTitle(id)

      triggerRepository
        .findByTemplateId(id, PageRequest.of(0, Int.MaxValue))
        .forEach { t =>
          logger.warn(s"Template id=[$id], title=['$templateTitle'] can't be deleted because it is used by trigger id=[${t.getId}], title=[${t.getTitle}]")
        }
    }
    val postCommitActions = ListBuffer.empty[PostCommitAction]
    templatesToDelete.foreach { id =>
      val template = releaseRepository.findById(id, new ResolveOptionsBuilder().withoutDecorators.build)
      logger.info(s"Removing orphaned template: ${template.getTitle}, with id ${template.getId}, from folder ${folderId}")
      releaseRepository.delete(id)
      postCommitActions.addOne(() => eventBus.publish(ReleaseDeletedEvent(template)))
    }

    ImportResult(List(CI.ids.withDeleted(templatesToDelete.toList)), postCommitActions.toList)
  }

  private def handleTriggers(folderId: String, idsToRetain: Set[String]): ImportResult = {
    val dbIds = triggerRepository.findByFolderId(folderId, nestedFolders = false, PageRequest.of(0, Int.MaxValue))
      .getContent
      .asScala
      .iterator
      .map(_.getId)
      .toSet
    val toDelete = dbIds.diff(idsToRetain)
    val postCommitActions = ListBuffer.empty[PostCommitAction]
    // we aren't actually deleting triggers here, but after the transaction
    // triggers require to be disabled and their actors to be shutdown before they can be deleted
    postCommitActions.addAll(toDelete.map(id => () => {
      logger.info(s"Removing orphaned trigger with id: ${id}, from folder ${folderId}")
      triggerService.deleteTrigger(id)
    }))
    ImportResult(List(CI.ids.withDeleted(toDelete.toList)), postCommitActions.toList)
  }

  private def handleConfigurations(folderId: String, idsToRetain: Set[String]): ImportResult = {
    val dbCisByIds = configurationRepository.findAllByTypeAndTitle[Configuration](typeOf[Configuration], title = null, folderId, folderOnly = true)
      .asScala
      .iterator
      .map(ci => ci.getId -> ci)
      .toMap
    val toDelete = dbCisByIds.removedAll(idsToRetain)
    val deletedConfigs = ListBuffer.empty[String]
    val postCommitActions = ListBuffer.empty[PostCommitAction]
    toDelete.foreachEntry { (id, ci) =>
      // don't try/catch ItemInUseException because that will still fail the whole transaction since REQUIRED propagation is used
      val releaseReferenced = releaseConfigurationReferencePersistence.isReferenced(id)
      val triggerReferenced = triggerConfigurationReferencePersistence.isReferenced(id)
      if (releaseReferenced || triggerReferenced) {
        logger.error(s"Configuration $id cannot be deleted because it is still used")
      } else {
        logger.info(s"Removing orphaned configuration: ${ci.getTitle}, with id ${ci.getId}, from folder ${folderId}")
        configurationRepository.delete(id)
        deletedConfigs.addOne(id)
        postCommitActions.addOne(() => eventBus.publish(ConfigurationDeletedEvent(ci)))
      }
    }

    ImportResult(List(CI.ids.withDeleted(deletedConfigs.toList)), postCommitActions.toList)
  }

  private def handleFolderVariables(folderId: String, idsToRetain: Set[String]): ImportResult = {
    val dbCisByIds = folderVariableRepository.getAllFromParent(folderId)
      .asScala
      .iterator
      .map(ci => ci.getId -> ci)
      .toMap
    val toDelete = dbCisByIds.removedAll(idsToRetain)
    val postCommitActions = ListBuffer.empty[PostCommitAction]
    toDelete.foreachEntry { (id, ci) =>
      logger.info(s"Removing orphaned folder variable: ${ci.getKey}, with id ${ci.getId}, from folder ${folderId}")
      folderVariableRepository.delete(id)
      postCommitActions.addOne(() => eventBus.publish(FolderVariableDeletedEvent(ci, null)))
    }

    ImportResult(List(CI.ids.withDeleted(toDelete.keys.toList)), postCommitActions.toList)
  }

  private def handlePatterns(folderId: String, idsToRetain: Set[String]): ImportResult = {
    val filters = new DeliveryPatternFilters()
    filters.setFolderId(folderId)
    val dbCisByIds = deliveryPatternService.searchPatterns(filters, Page(0, Int.MaxValue, 0))
      .asScala
      .iterator
      .map(ci => ci.getId -> ci)
      .toMap
    val toDelete = dbCisByIds.removedAll(idsToRetain)
    val postCommitActions = ListBuffer.empty[PostCommitAction]
    toDelete.foreachEntry { (id, ci) =>
      logger.info(s"Removing orphaned delivery pattern: ${ci.getTitle}, with id ${ci.getId}, from folder ${folderId}")
      deliveryRepository.delete(id)
      postCommitActions.addOne(() => eventBus.publish(DeliveryDeletedEvent(ci)))
    }

    ImportResult(List(CI.ids.withDeleted(toDelete.keys.toList)), postCommitActions.toList)
  }

  private def handleDashboards(folderId: String, idsToRetain: Set[String]): ImportResult = {
    val dbCisByIds = dashboardService.search(folderId, enforcePermissions = false)
      .iterator
      .map(ci => ci.getId -> ci)
      .toMap
    val toDelete = dbCisByIds.removedAll(idsToRetain)
    val postCommitActions = ListBuffer.empty[PostCommitAction]
    toDelete.foreachEntry { (id, ci) =>
      // N.B. this is for folder dashboards only for now, so DashboardSecurity#clearPermissions isn't needed
      logger.info(s"Removing orphaned dashboard: ${ci.getTitle}, with id ${ci.getId}, from folder ${folderId}")
      sqlDashboardRepository.deleteDashboard(id)
      postCommitActions.addOne(() => eventBus.publish(DashboardDeletedEvent(ci)))
    }

    ImportResult(List(CI.ids.withDeleted(toDelete.keys.toList)), postCommitActions.toList)
  }

  private def handleNotifications(folderId: String, idsToRetain: Set[String]): ImportResult = {
    // we just check whether it's empty or not. you can't really get rid of notification settings
    // so orphaning them means resetting them to inherit parent config
    if (idsToRetain.isEmpty) {
      // the cache will get global notification settings if it doesn't find one on folder, so avoid getting from cache
      val maybeNotificationSettings = configurationRepository
        .findAllByTypeAndTitle[EmailNotificationSettings](typeOf[EmailNotificationSettings], null, folderId, folderOnly = true)
        .asScala
        .headOption

      maybeNotificationSettings.fold(ImportResult.empty) { settings =>
        logger.info(s"Removing orphaned notification: ${settings.getTitle}, with id ${settings.getId}, from folder ${folderId}")
        configurationRepository.delete(settings.getId)
        ImportResult(
          List(CI.ids.withDeleted(settings.getId)),
          List(() => eventBus.publish(ConfigurationDeletedEvent(settings)))
        )
      }
    } else {
      ImportResult.empty
    }
  }
}
