const defaultSize = {
    width: 1280,
    height: 800
};

class Browser {
    static open() {
        browser.get('#/');
        return this;
    }

    static setDefaultSize() {
        return this.setSize(defaultSize.width, defaultSize.height);
    }

    static setSize(width, height) {
        const resize = (windowWidth, windowHeight) =>
            window.resizeTo(windowWidth, windowHeight);
        this.executeVoidScript(resize, width, height);
    }

    static expectUrlToBe(url) {
        expect(browser.getCurrentUrl()).toContain(url);
        return this;
    }
    static expectUrlToContain = Browser.expectUrlToBe;

    static expectUrlParams(params) {
        const parseToUrl = params => Object.entries(params).map(([key, value]) => {
            if (_.isBoolean(value))
                return key;
            else if (_.isArray(value))
                return value.map(e => `${key}=${e}`).join('&');
            else
                return `${key}=${value}`;
        }).join('&');

        const paramString = `?` + parseToUrl(params);

        return this.wait(EC.urlContains(paramString), 'waiting for URL params to change');
    }

    static wait(conditionFunction, message, timeout = 30000) {
        return browser.wait(conditionFunction, timeout, message);
    }

    static waitFor(cssSelector, timeout) {
        return this.wait(EC.presenceOf(element(By.$(cssSelector))), `waiting for '${cssSelector}' to be present`, timeout);
    }

    static waitForDisplayed(cssSelector, timeout) {
        return this.wait(EC.visibilityOf(element(By.$(cssSelector))), `waiting for '${cssSelector}' to be present && displayed`, timeout);
    }

    static scrollTo(locator) {
        let scrollIntoView = function () {
            return arguments[0].scrollIntoView(false);
        };
        return this.executeVoidScript(scrollIntoView, browser.findElement(locator));
    }

    static executeVoidScript(scriptFunction, ...scriptArguments) {
        const script = `(${scriptFunction}).apply(null, arguments);`;
        return browser.executeScript(script, ...scriptArguments);
    }

    static hover(locator) {
        return browser.actions().mouseMove(element(locator)).perform();
    }

    static blurActiveElement() {
        return this.executeVoidScript(() => document.activeElement.blur());
    }

    static clickElseWhere() {
        return this.executeVoidScript(() => {
            $(document).mousedown();
            $(document).click();
        });
    }

    static softReload() {
        let remoteScript = function () {
            let state = angular.element('body').injector().get('$state');
            return state.reload();
        };

        browser.sleep(500); // to provide running requests with some time to finish
        Browser.executeVoidScript(remoteScript);
    }

    // used to isolate tests with download bar in separate window
    static switchToNewWindow() {
        const script = `window.open('#/', '_blank', 'menubar=0,width=${defaultSize.width},height=${defaultSize.height}')`;
        browser.executeScript(script).then(() => {
            browser.getAllWindowHandles().then((handles) => {
                const newWindow = handles[1];
                browser.switchTo().window(newWindow);
            });
        });
    }

    static backToOriginalWindow() {
        browser.executeScript('window.close()').then(() => {
            browser.getAllWindowHandles().then((handles) => {
                const originalWindow = handles[0];
                browser.switchTo().window(originalWindow);
            });
        });
    }
}

global.Browser = Browser;
