class CalendarPage {
    constructor() {
        Browser.waitFor('#calendar');
    }

    toggleNavigator() {
        element(By.id('navigator-toggler')).click();
        return this;
    }

    toggleFilters() {
        element(By.css('#calendar .dropdown-button')).click();
        return this;
    }

    searchBy(criteria) {
        element(By.css('#searchFilter')).clear();
        element(By.css('#searchFilter')).sendKeys(criteria);
        browser.sleep(1000);
        return this;
    }

    openReleaseTooltip(title) {
        element(By.$(`.release:contains('${title}'):first`)).click();
        browser.sleep(2000); // wait for animation
        return this;
    }

    showNextMonth() {
        element(By.css('.button.next-month')).click();
        return this;
    }

    showPreviousMonth() {
        element(By.css('.button.previous-month')).click();
        return this;
    }

    openDayModal(dayNumber) {
        this.openMenuItem(dayNumber, 'label');
        return new CalendarDayModal();
    }

    setBlackout(label, startDate, endDate) {
        this.openMenuItem('10', 'Set blackout period');
        let blackout = new CalendarBlackoutEditModal();
        blackout.setLabel(label);
        blackout.setStartDate(startDate);
        blackout.setEndDate(endDate);
        blackout.save();
        return this;
    }

    removeBlackout(label) {
        element.all(By.$(`.blackout-label:contains('${label}')`)).first().click();
        new CalendarBlackoutDeleteModal().del();
        return this;
    }

    openBlackoutTooltip(label) {
        Browser.waitForDisplayed(`.blackout-label:contains('${label}')`);
        element.all(By.$(`.blackout-label:contains('${label}')`)).first().click();
        return new CalendarBlackoutTooltip();
    }

    openMenuItem(dayNumber, item) {
        element(By.$(`.day:contains('${dayNumber}') .context-menu-icon`)).click();
        element(By.$(`xlr-context-menu a:contains('${item}')`)).click();
    }

    expectFirstDayOfWeekToBe(firstDayOfWeek) {
        expect(element(By.$(`#days-name > div:first-child:contains("${firstDayOfWeek}")`))).toBePresent();
        return this;
    }

    expectBlackoutToBePresent(label) {
        expect(element.all(By.$(`.blackout .blackout-label:contains("${label}")`)).count()).toBeGreaterThan(0);
        return this;
    }

    expectBlackoutToBeAbsent(label) {
        expect(element.all(By.$(`.blackout .blackout-label:contains("${label}")`)).count()).toBe(0);
        return this;
    }

    expectBlackoutLabelToBe(label) {
        expect(element.all(By.$(`.blackout-label:contains("${label}")`)).first().isPresent()).toBe(true);
        return this;
    }
}

class CalendarDayModal extends Modal {

    getColorPicker() {
        return new ColorPicker(element(By.$('.modal-dialog .pick-color')));
    }

    setTitle(text) {
        element(By.model('$ctrl.editedDay.label')).clear().sendKeys(text);
        return this;
    }

    save() {
        return element(By.$('.modal-dialog .button.save')).click();
    }
}

class CalendarBlackoutEditModal extends Modal {
    setLabel(text) {
        element(By.$('.modal-dialog #label')).clear().sendKeys(text);
        return this;
    }

    setStartDate(date) {
        this.setDate('.start-date .date-editor', date);
        return this;
    }

    setEndDate(date) {
        this.setDate('.end-date .date-editor', date);
        return this;
    }

    setDate(selector, date) {
        let dateTimePicker = new DateTimePicker(selector);
        dateTimePicker.setDate({day: date.date(), month: date.month() + 1, year: date.year()});
        dateTimePicker.setTime({hour: date.hour(), minute: date.minute()}, () => element(By.$('.modal-dialog #label')).click());
    }

    expectTitleToBe(label) {
        expect(element(By.$('.modal-dialog .modal-header')).getText()).toContain(`Edit ${label}`);
        return this;
    }

    expectLabelToBe(label) {
        expect(element(By.$('.modal-dialog input')).getAttribute('value')).toBe(label);
        return this;
    }

    expectStartDateToBe(startDate) {
        Dates.expectInputContainingDate(By.$('.start-date .date-editor .date input'), 'shortDate', startDate);
        Dates.expectInputContainingDate(By.$('.start-date .date-editor .time input'), 'shortTime', startDate);
        return this;
    }

    expectEndDateToBe(endDate) {
        Dates.expectInputContainingDate(By.$('.end-date .date-editor .date input'), 'shortDate', endDate);
        Dates.expectInputContainingDate(By.$('.end-date .date-editor .time input'), 'shortTime', endDate);
        return this;
    }

    save() {
        element(By.$('.modal-dialog .button.save')).click();
    }

    cancel() {
        element(By.$('.modal-dialog .button.cancel')).click();
    }
}

class CalendarBlackoutDeleteModal extends Modal {
    del() {
        element(By.$('.modal-dialog .button.continue')).click();
    }
}

class CalendarBlackoutTooltip {
    constructor() {
        Browser.waitFor('.popover-blackout');
    }

    openEdit() {
        browser.actions().mouseMove(element(By.$('#days-name'))).perform();
        browser.sleep(300);
        element(By.$('.link.edit')).click();
        return new CalendarBlackoutEditModal();
    }

    openDelete() {
        // because black-ish tooltip is in the way between pointer and 'delete' link.
        browser.actions().mouseMove(element(By.$('#days-name'))).perform();
        // way is free, click it!
        browser.sleep(300);
        element(By.$('.link.delete')).click();
        return new CalendarBlackoutDeleteModal();
    }

    close() {
        element(By.$('.popover-close')).click();
        return new CalendarPage();
    }

    expectLabelToBe(label) {
        expect(element(By.$('.popover-title > span')).getText()).toBe(label);
        return this;
    }

    expectStartDateToBe(startDate) {
        Dates.expectContainingDate(By.$('.popover-content .start-date'), 'short', startDate);
        return this;
    }

    expectEndDateToBe(endDate) {
        Dates.expectContainingDate(By.$('.popover-content .end-date'), 'short', endDate);
        return this;
    }
}

global.CalendarPage = CalendarPage;
global.CalendarDayModal = CalendarDayModal;
