class DashboardPage {
    constructor() {
        Browser.waitFor('.xlr-dashboard');
        browser.sleep(500);
    }

    static openDashboardOf(releaseId) {
        browser.setLocation(`/releases/${releaseId}/summary`);
        return new DashboardPage();
    }

    expectEditContentsButtonDisplayed(isDisplayed) {
        expect(element(By.$("#release-header .button:contains('Configure')"))).toBePresent(isDisplayed);
        return this;
    }

    expectViewMode() {
        expect(element(By.$("#release-header .button.primary:contains('Configure')"))).toBeDisplayed();
        return this;
    }

    expectDesignMode() {
        expect(element(By.$("#release-header .button:contains('Back to view mode')"))).toBeDisplayed();
        return this;
    }

    editContents() {
        element(By.$("#release-header .button.primary:contains('Configure')")).click();
        return this;
    }

    addTile(type) {
        element(By.$("#release-header .button:contains('Configure dashboard')")).click();
        element(By.$(`.tile-card-title:contains(${type}) .primary`)).click();
        return this;
    }

    save() {
        element(By.$("#release-header .button:contains('Back to view mode')")).click();
        return this;
    }

    getTile(content) {
        return new Tile(content);
    }

    expectNoTile(content) {
        expect(element(By.$(`.xlr-tile-container:contains('${content}')`))).toBePresent(false);
        return this;
    }

    expectTile(content) {
        expect(element(By.$(`.xlr-tile-container:contains('${content}')`))).toBePresent(true);
        return this;
    }
}

let precisionDeltaPx = undefined;
class Tile {
    static initClass() {

        precisionDeltaPx = 20;
    }

    constructor(content) {
        this.tileLocator = () => By.$(`.gridster-item:contains('${content}')`);
        this.tile = function () {
            return element(this.tileLocator());
        };
        expect(this.tile()).toBeDisplayed();
        this.clickButtonByTooltip = tooltip => {
            this.tile().element(By.$(`.panel-heading span[tooltip="'${tooltip}'"]`)).click();
            return this;
        };
        this;
    }

    expectButtonDisplayed(tooltip, isDisplayed) {
        expect(this.tile().element(By.$(`.panel-heading span[tooltip="'${tooltip}'"]`))).toBePresent(isDisplayed);
        return this;
    }

    expectDeleteButtonDisplayed(isDisplayed) {
        return expectButtonDisplayed('Delete', isDisplayed);
    }

    expectResizeHandleDisplayed(isDisplayed) {
        Browser.hover(this.tileLocator());
        expect(this.tile().element(By.$('.gridster-item-resizable-handler'))).toBePresent(isDisplayed);
        return this;
    }

    expectTitle(title) {
        expect(this.tile().element(By.$(`.panel-heading:contains('${title}')`))).toBeDisplayed();
        return this;
    }

    expectContent(text) {
        expect(this.tile().element(By.$(".panel-body")).getText()).toContain(text);
        return this;
    }

    expectIframeContent(cssSelector, text) {
        let script = `return $("iframe").contents().find('${cssSelector}').get()[0].innerHTML`;
        browser.executeScript(script).then((content) => {
            expect(content).toContain(text);
        });
        return this;
    }

    openDetailsView() {
        this.clickButtonByTooltip('View details');
        return new TileDetails();
    }

    configure() {
        browser.waitForAngular();
        this.clickButtonByTooltip('Configure');
        return new TileConfiguration(this.tile);
    }

    save() {
        return this.clickButtonByTooltip('Save');
    }

    cancel() {
        return this.clickButtonByTooltip('Cancel');
    }

    delete() {
        this.clickButtonByTooltip('Delete');
        return new TileDeleteDialog(this.tile);
    }

    expectSize(columns, rows) {
        this.tile().getSize().then(size => {
                return this.tile().element(By.xpath('../..')).getSize().then(function (dashboardSize) {
                    let columnWidthPx = dashboardSize.width / 3;
                    let rowHeightPx = 75;
                    let widthBoundaries = [(columns * columnWidthPx) - precisionDeltaPx, (columns * columnWidthPx) + precisionDeltaPx];
                    let heightBoundaries = [(rows * rowHeightPx) - precisionDeltaPx, (rows * rowHeightPx) + precisionDeltaPx];
                    expect(widthBoundaries[0] <= size.width && size.width <= widthBoundaries[1]).toBe(
                        true, `Tile size width does not match boundaries ${widthBoundaries}: ${size.width} (expected ${columns} columns)`);
                    expect(heightBoundaries[0] <= size.height && size.height <= heightBoundaries[1]).toBe(
                        true, `Tile size height does not match boundaries ${heightBoundaries}: ${size.height} (expected ${rows} rows)`);
                });
            }
        );
        return this;
    }

    resize(plusColumn, plusRow) {
        Browser.hover(this.tileLocator());
        let seHandle = this.tile().element(By.$('.gridster-item-resizable-handler.handle-se'));
        browser.actions()
            .mouseMove(seHandle.getWebElement(), {x: 2, y: 2})
            .mouseDown()
            // nudge it a little bit in given directly, that is enough for gridster resizing to trigger
            .mouseMove(seHandle.getWebElement(), {
                x: plusColumn * 20,
                y: plusRow * 20
            })
            .mouseUp()
            .perform();
        // wait for animation
        browser.sleep(500);
        return this;
    }

    expectInFirstColumn(isInFirstColumn) {
        this.tile().getCssValue('left').then(function (leftPx) {
            let leftOffset = parseInt(leftPx.replace('px', '')) - 10;
            if (isInFirstColumn) {
                expect(leftOffset < 20).toBe(true, `Expected tile to be in the first column, but its left offset was ${leftPx}`);
            } else {
                expect(leftOffset > 20).toBe(true, `Expected tile to be in the second or righter column, but its left offset was ${leftPx}`);
            }
        });
        return this;
    }

    dragRight() {
        let moveTileButton = this.tile().element(By.$('.xlr-tile-drag-handle'));
        browser.actions()
            .mouseMove(moveTileButton.getWebElement(), {x: 2, y: 2})
            .mouseDown()
            // nudge it alittle bit in given directly, that is enough for gridster resizing to trigger
            .mouseMove(moveTileButton.getWebElement(), {x: 2 + 300, y: 2})
            .mouseUp()
            .perform();
        // wait for animation
        browser.sleep(500);
        return this;
    }
}
Tile.initClass();


class TileDetails {

    expectContent(text) {
        expect(element(By.$(".dashboard-details-view")).getText()).toContain(text);
        return this;
    }

    closeDetailsView() {
        element(By.$(".dasboardsummary.button:visible")).click();
        return this;
    }
}


class TileConfiguration extends Modal {
    constructor(tile) {
        super(...arguments);
        this.tile = tile;
        this.tileConfiguration = element(By.$('.xlr-tile-config-modal'));
    }

    cancel() {
        this.tileConfiguration.element(By.$("div.cancel:contains('Cancel')")).click();
        return this;
    }

    save() {
        this.tileConfiguration.element(By.$("div.primary:contains('Save')")).click();
        return this;
    }

    setProperty(name, value) {
        browser.waitForAngular();
        this.tileConfiguration.element(By.$(`input[name=${name}]`)).clear();
        this.tileConfiguration.element(By.$(`input[name=${name}]`)).sendKeys(value);
        return this;
    }

    expectContent(text) {
        expect(this.tileConfiguration.element(By.$(`.modal-body:contains('${text}')`))).toBeDisplayed();
        return this;
    }

    groupPath(field) {
        return `#modal #${field}`;
    }

    editorPath(field) {
        return `#modal #${field} .field`;
    }

    getValue(field) {
        return new InlineEditor(this.editorPath(field)).value();
    }

    expectValue(field, value) {
        expect(this.getValue(field)).toBe(value);
        return this;
    }

    setTextField(field, value) {
        new InlineEditor(this.editorPath(field)).set(value);
        return this;
    }

    getStringMapField(field) {
        return new XlrMapStringStringWithVariables(this.groupPath(field));
    }

    getStringListField(field) {
        return new XlrListStringWithVariables(this.groupPath(field));
    }

    getPasswordField(field) {
        return new XlrInlinePasswordWithVariables(this.groupPath(field));
    }

    getIntegerField(field) {
        return new XlrIntegerWithVariables(this.groupPath(field));
    }

    getIntegerField(field) {
        return new XlrIntegerWithVariables(this.groupPath(field));
    }

    getBooleanField(field) {
        return new XlrBooleanWithVariables(this.groupPath(field));
    }
}

class TileDeleteDialog extends Modal {
    constructor(tile) {
        super(...arguments);
        this.tile = tile;
        this.tileDeleteDialog = element(By.$('.xlr-tile-delete-dialog'));
    }

    cancel() {
        this.tileDeleteDialog.element(By.$("button.cancel:contains('Cancel')")).click();
        return this;
    }

    delete() {
        this.tileDeleteDialog.element(By.$("button.primary:contains('Delete')")).click();
        return this;
    }
}

global.DashboardPage = DashboardPage;
global.Tile = Tile;
global.TileConfiguration = TileConfiguration;
global.TileDeleteDialog = TileDeleteDialog;
