let JIRA_ISSUE_API_URL = "https://xebialabs.atlassian.net/rest/api/2/issue";

let configurationItemsIds = undefined;
let usernames = undefined;
let releaseIds = undefined;
let archivedReleaseIds = undefined;
let globalVariableIds = undefined;
class Fixtures {
    static initClass() {
        configurationItemsIds = [];
        usernames = [];
        releaseIds = [];
        archivedReleaseIds = [];
        globalVariableIds = [];
    }

    static sendRequest(type, url, data) {
        return Flow.execute((fulfill, reject) =>
            Server.doRequest(type, url, data).then(fulfill, error => reject(error))
        );
    }

    static sendRequestAndIgnoreResult(type, url, data) {
        return Flow.execute(fulfill => Server.doRequest(type, url, data).then(fulfill, fulfill));
    }

    static deleteCI(id) {
        return this.sendRequest('DELETE', `repository/ci/${id}`);
    }

    static release(release) {
        initReleaseDefaults(release);
        releaseIds.push(release.id);
        return this.sendRequest('POST', 'fixtures/release', release);
    }

    static archivedRelease(release) {
        initReleaseDefaults(release);
        releaseIds.push(release.id);
        return this.sendRequest('POST', 'fixtures/archived', release);
    }

    static ci(ci) {
        configurationItemsIds.push(ci.id);
        return this.sendRequest('POST', 'fixtures', [ci]);
    }

    static cis(cis) {
        for (let ci of cis) {
            configurationItemsIds.push(ci.id);
        }
        return this.sendRequest('POST', 'fixtures', cis);
    }

    static deleteRelease(releaseId) {
        if (releaseId.indexOf('Applications/') === -1) {
            releaseId = 'Applications/' + releaseId;
        }
        return this.sendRequest('DELETE', `fixtures/${releaseId}`);
    }

    static folder(folder) {
        configurationItemsIds.push(folder.id);
        return this.sendRequest('POST', 'fixtures', getFolderEntities(folder));
    }

    static roles(rolePrincipals) {
        return this.sendRequest('POST', 'fixtures/roles/principals', rolePrincipals);
    }

    static permissions(rolePermissions) {
        return this.sendRequest('POST', 'fixtures/roles/permissions/global', rolePermissions);
    }

    static addUser(username, password) {
        usernames.push(username);
        return this.sendRequest('POST', 'fixtures/user', {username, password});
    }

    static deleteUser(username) {
        return this.sendRequest('DELETE', `fixtures/user/${username}`);
    }

    static taskAccesses(taskAccesses) {
        return this.sendRequest('PUT', 'tasks/types-access', taskAccesses);
    }

    static deleteTaskAccesses() {
        return this.sendRequest('DELETE', "fixtures/task-access");
    }

    static addDefaultUser() {
        return this.addUser('Itchy', 'Itchy');
    }

    static addUserProfile(username, profile) {
        if (!profile) {
            profile = {};
        }
        profile.id = `Configuration/Users/${username}`;
        profile.type = 'xlrelease.UserProfile';
        profile.canonicalId = username.toLowerCase();
        configurationItemsIds.push(profile.id);
        return this.sendRequest('POST', 'fixtures', [profile]);
    }

    static deleteUserProfile(username) {
        const url = 'fixtures/cis';
        const profile = {
            type: 'xlrelease.UserProfile',
            id: 'Configuration/Users/' + username
        };
        return this.sendRequest('DELETE', url, [profile]);
    }

    static updateUserProfile(username, profile) {
        profile.id = 'Configuration/Users/' + username;
        profile.type = 'xlrelease.UserProfile';
        profile.canonicalId = username.toLowerCase();

        return this.sendRequest('PUT', 'fixtures', [profile]);
    }

    static resetDefaultUserProfile() {
        this.updateUserProfile('Itchy', {email: 'itchy@default.com', loginAllowed: true});
    };

    static expectUserProfileExists(username, userEmail, fullName) {
        return this.sendRequest('GET', `fixtures/userProfile?name=${username}&email=${userEmail}&fullName=${fullName}`);
    }

    static expectContainingAttachments(releaseId, tempFile) {
        let ok = function () {
        };
        let failure = function (message) {
            throw `Can't find attachment '${tempFile.getName()}' with content : '${tempFile.getContent()}'. Message from server: '${message}'`;
        };
        if (releaseId.indexOf('Applications/') === -1) {
            releaseId = 'Applications/' + releaseId;
        }
        return this.sendRequest('POST', `fixtures/expectContainingAttachments/${releaseId}`, {name: tempFile.getName(), expectedContent: tempFile.getContent()})
            .then(ok, failure);
    }

    static addJiraTask() {
        return Flow.execute(function (fulfill, reject) {
            let data = {
                fields: {
                    project: {key: "RIT"},
                    summary: "Test Task",
                    description: "Test task description",
                    issuetype: {name: "Task"}
                }
            };
            let options = {
                method: "POST",
                body: data,
                uri: JIRA_ISSUE_API_URL,
                json: true,
                auth: {username: 'xlr-jira-testuser@xebialabs.com', password: '9Es3MR8pnRyf', sendImmediately: true},
                resolveWithFullResponse: true
            };
            return requestPromise(options).then(result => fulfill(result.body["key"])).catch(result => reject(result.error));
        });
    }

    static expectJiraTaskStatus(taskId, expectedStatus) {
        return Flow.execute(function (fulfill, reject) {
            let options = {
                method: "GET",
                uri: `${JIRA_ISSUE_API_URL}/${taskId}`,
                json: true,
                auth: {username: 'xlr-jira-testuser@xebialabs.com', password: '9Es3MR8pnRyf', sendImmediately: true},
                resolveWithFullResponse: true
            };
            return requestPromise(options).then(function (result) {
                let {body} = result;
                let status = body.fields.status.name;
                if (status !== expectedStatus) {
                    reject(`Expected task ${taskId} to be '${expectedStatus}' but was '${status}'. Message from server : ${body}`);
                }
                return fulfill();
            }).catch(result => reject(result.error));
        });
    }

    static expectJiraTaskSummary(taskId, expectedSummary) {
        return Flow.execute(function (fulfill, reject) {
            let options = {
                method: "GET",
                uri: `${JIRA_ISSUE_API_URL}/${taskId}`,
                json: true,
                auth: {username: 'xlr-jira-testuser@xebialabs.com', password: '9Es3MR8pnRyf', sendImmediately: true},
                resolveWithFullResponse: true
            };
            return requestPromise(options).then(function (result) {
                let {body} = result;
                let {summary} = body.fields;
                if (summary !== expectedSummary) {
                    reject(`Expected task ${taskId} summary to be '${expectedSummary}' but was '${summary}'. Message from server : ${body}`);
                }
                return fulfill();
            }).catch(result => reject(result.error));
        });
    }


    static globalVariable(variable) {
        variable.id = variable.id || `Configuration/variables/global/Variable_${variable.key}`;
        globalVariableIds.push(variable.id);
        variable.requiresValue = false;
        variable.showOnReleaseStart = false;
        return this.sendRequest('POST', 'api/v1/config/Configuration/variables/global', variable);
    }

    static deleteGlobalVariable(variableId) {
        return this.sendRequestAndIgnoreResult('DELETE', `api/v1/config/${variableId}`);
    }

    static deleteAllGlobalVariables() {
        return this.sendRequest('DELETE', "fixtures/globalVariables");
    }

    static clearCalendar() {
        return this.sendRequest('DELETE', "fixtures/calendar");
    }

    static addBlackout(name, from, to) {
        return this.sendRequest('POST', 'calendar/blackouts', {
            label: name,
            startDate: from.toDate(),
            endDate: to.toDate(),
            type: "xlrelease.Blackout",
            id: null
        });
    }

    static clean() {
        for (let username of usernames) {
            this.deleteUser(username);
        }
        for (let releaseId of releaseIds.reverse()) {
            this.deleteRelease(releaseId);
        }
        for (let configurationItemId of configurationItemsIds) {
            this.deleteCI(configurationItemId);
        }
        for (let variableId of globalVariableIds) {
            this.deleteGlobalVariable(variableId);
        }
        releaseIds = [];
        configurationItemsIds = [];
        usernames = [];
        return globalVariableIds = [];
    }
}
Fixtures.initClass();

global.fixtures = () => Fixtures;
