class VariablesPage {
    constructor(elementId) {
        Browser.waitFor(elementId);
    }

    setSearchFilter(filterText) {
        element(By.model('filters.filter')).clear();
        element(By.model('filters.filter')).sendKeys(filterText);
        return this;
    }

    edit(name) {
        element(By.$(`.variable:contains('${name}') .action.edit`)).click();
        return new VariableDetails();
    }

    delete(name) {
        element(By.$(`.variable:contains('${name}') .action.delete`)).click();
        return new VariableDeleteModal();
    }

    drag(from, to) {
        let fromElement = element(By.$(`.variable:contains('${from}')`));
        let toElement = element(By.$(`.variable:contains('${to}')`));
        browser.actions().mouseMove(fromElement.getWebElement(), { x: 0, y: 0 }).mouseDown().mouseMove(toElement.getWebElement()).mouseUp().perform();
        return this;
    }

    newVariable() {
        element(By.$('.button.new-variable')).click();
        return new VariableDetails();
    }

    expectNewVariableButtonDisplayed(expected = true) {
        expect(element(By.$('.button.new-variable'))).toBePresent(expected);
        return this;
    }

    expectEditVariableButtonsEnabled(expected = true) {
        expect(element(By.$('.variable .action.edit:enabled'))).toBePresent(expected);
        return this;
    }

    expectDeleteVariableButtonsEnabled(expected = true) {
        expect(element(By.$('.variable .action.delete:enabled'))).toBePresent(expected);
        return this;
    }

    expectVariableDisplayed(variableName) {
        expect(element(By.$(`.variable:contains('${variableName}')`))).toBePresent();
        return this;
    }

    expectVariableNotDisplayed(variableName) {
        expect(element(By.$(`.variable:contains('${variableName}')`))).not.toBePresent();
        return this;
    }

    expectVariablesDisplayed(variables) {
        for (let name in variables) {
            let expected = variables[name];
            expect(element(By.$(`.variable:contains('${name}')`))).toBePresent(expected);
        }
        return this;
    }

    expectActionsDisplayed(actions) {
        for (let name in actions) {
            let expected = actions[name];
            expect(element(By.$(`.action:contains('${name}')`))).toBePresent(expected);
        }
        return this;
    }

    expectVariablesDisplayedInOrder(variableNames) {
        element.all(By.css('.variable > div > div:first-child')).then(function (elements) {
            expect(elements.length).toBe(variableNames.length);
            return (() => {
                let result = [];
                for (let [name, item] of _.zip(variableNames, elements)) {
                    let item1;
                    if (item) {
                        item1 = expect(item.getText()).toContain(name);
                    }
                    result.push(item1);
                }
                return result;
            })();
        });
        return this;
    }

    expectVariablesCountToBe(expectedCount) {
        expect(element.all(By.css('.variable')).count()).toBe(expectedCount);
        return this;
    }

    expectVariablesCountToBeAtLeast(expectedCount) {
        expect(element.all(By.css('.variable')).count()).toBeGreaterThanOrEqual(expectedCount);
        return this;
    }

    expectVariableWithNameValueAndType(name, value, type) {
        expect(element(By.$(`.variable:contains('${name}'):contains('${value}'):contains('${type}')`)).isPresent()).toBe(
            true,
            `Expected variable '${name}' with value '${value}' of '${type}'`,
        );
        return this;
    }

    expectDateVariableWithNameAndValue(name, value, format) {
        Dates.expectContainingDate(By.$(`.variable:contains('${name}'):contains('Date') > div > div:nth-child(3)`), format, value);
        return this;
    }

    expectVariableWithLabel(label) {
        expect(element(By.$(`.variable:contains('${label}')`)).isPresent()).toBe(true, `Expected variable with label '${label}'`);
        return this;
    }

    expectHeader(label) {
        expect(element(By.$(`.variables-list .header-row:contains('${label}')`))).toBePresent();
        return this;
    }

    refresh() {
        Browser.softReload();
        return this;
    }
}

class VariableDeleteModal extends Modal {
    constructor() {
        super(...arguments);
        this.variableModal = element(By.id('modal'));
    }

    save() {
        this.variableModal.element(By.css('button.delete')).click();
        Browser.wait(() =>
            element(By.id('modal'))
                .isDisplayed()
                .then((isDisplayed) => !isDisplayed),
        );
        return this;
    }

    expectReplacementPromptDisplayed(visible) {
        expect(this.variableModal.element(By.css('div.variable-replacement')).isPresent()).toBe(visible);
        return this;
    }

    expectReplacementValue(value) {
        let editor = new InlineEditor("#modal .form-group:contains('Replace') div[inline-text-editor]");
        expect(editor.value()).toBe(value);
    }

    setReplacementValue(value) {
        let editor = new InlineEditor("#modal .form-group:contains('Replace') div[inline-text-editor]");
        editor.set(value);
        return this;
    }
}

class VariableDetails extends Modal {
    constructor() {
        super(...arguments);
        this.variableDetails = element(By.id('modal'));
    }

    setName(name) {
        this.variableDetails.element(By.css('.variable-name input')).clear();
        this.variableDetails.element(By.css('.variable-name input')).sendKeys(name);
        return this;
    }

    selectType(type) {
        this.variableDetails.element(By.$(`option[label=\"${type}\"]`)).click();
        return this;
    }

    setTextValue(value) {
        this.variableDetails.element(By.model('var.value')).clear();
        this.variableDetails.element(By.model('var.value')).sendKeys(value);
        return this;
    }

    addPossibleValue(value) {
        this._possibleValues().editor().enter(value).add();
        return this;
    }

    choosePossibleValuesVariable(name) {
        this._possibleValues().toggleMode().variableSelector().focus().select(name);
        return this;
    }

    createPossibleValuesVariable(name) {
        this._possibleValues().toggleMode().variableSelector().focus().enter(name).select(name);
        return this;
    }

    expectPossibleValuesVariable(name) {
        this._possibleValues().expectVariableMode().variableSelector().expectValue(name);
        return this;
    }

    selectValue(value) {
        this.variableDetails.element(By.$(`option:contains('${value}')`)).click();
        return this;
    }

    clickCheckbox() {
        this.variableDetails.element(By.model('var.value')).click();
        return this;
    }

    setLabel(value) {
        this.variableDetails.element(By.model('var.label')).clear();
        this.variableDetails.element(By.model('var.label')).sendKeys(value);
        return this;
    }

    setDescription(value) {
        this.variableDetails.element(By.model('var.description')).clear();
        this.variableDetails.element(By.model('var.description')).sendKeys(value);
        return this;
    }

    save() {
        browser.sleep(100);
        this.variableDetails.element(By.css('button.save')).click();
        Browser.wait(() =>
            element(By.id('modal'))
                .isDisplayed()
                .then((isDisplayed) => !isDisplayed),
        );
        return this;
    }

    cancel() {
        this.variableDetails.element(By.css('button.cancel')).click();
        return this;
    }

    close() {
        return element(By.$('.close')).click();
    }

    expectSaveButtonInvisible() {
        expect(this.variableDetails.all(By.$('button.save:visible')).count()).toEqual(0);
        return this;
    }

    expectSaveButtonDisabled() {
        expect(this.variableDetails.all(By.$('button.save:disabled')).count()).toEqual(1);
        return this;
    }

    expectSaveButtonEnabled() {
        expect(this.variableDetails.all(By.$('button.save:enabled')).count()).toEqual(1);
        return this;
    }

    expectLabel(label) {
        expect(this.variableDetails.element(By.$(`.control-label:contains('${label}')`))).toBePresent();
        return this;
    }

    expectValueToBe(value) {
        this.variableDetails
            .element(By.css('.variable-value'))
            .getText()
            .then((text) => {
                if (text) {
                    expect(text).toContain(value);
                } else {
                    expect(this.variableDetails.element(By.model('var.value')).getAttribute('value')).toBe(value);
                }
            });
        return this;
    }

    expectNotToBeEditable(value) {
        expect(this.variableDetails.element(By.$(`.variable-name span:contains('${value}')`))).toBePresent();
    }

    expectName(value) {
        expect(this.variableDetails.element(By.$(`.variable-name:contains('${value}')`))).toBePresent();
        return this;
    }

    expectNameInput(value) {
        expect(this.variableDetails.element(By.model('var.key')).getAttribute('value')).toBe(value);
        return this;
    }

    expectLabelAndDescription(label, desc) {
        expect(this.variableDetails.element(By.model('var.label')).getAttribute('value')).toBe(label);
        expect(this.variableDetails.element(By.model('var.description')).getAttribute('value')).toBe(desc);
        return this;
    }

    expectRequired(value) {
        expect(this.variableDetails.element(By.model('var.requiresValue')).getAttribute('value')).toBe(value ? 'on' : 'off');
        return this;
    }

    expectRequiredFieldNotPresent() {
        expect(this.variableDetails.element(By.model('var.requiresValue'))).toBePresent(false);
        return this;
    }

    expectShowOnReleaseStart(value) {
        expect(this.variableDetails.element(By.model('var.showOnReleaseStart')).getAttribute('value')).toBe(value ? 'on' : 'off');
        return this;
    }

    expectShowOnReleaseStartFieldNotPresent() {
        expect(this.variableDetails.element(By.model('var.showOnReleaseStart'))).toBePresent(false);
        return this;
    }

    expectNoValidationErrors() {
        expect(this.variableDetails.element(By.$('.field-errors span:visible'))).toBePresent(false);
        return this;
    }

    expectValidationError(text) {
        expect(this.variableDetails.element(By.$(`.field-errors span:visible:contains('${text}')`))).toBePresent();
        return this;
    }

    expectPossibleValues(values) {
        this._possibleValues().editor().shouldHaveItems(values);
        return this;
    }

    setDateTime(date) {
        let dateTimePicker = new DateTimePicker(`#modal .date-editor:first`);
        dateTimePicker.setDateTime(date);
        return this;
    }

    _possibleValues() {
        return new XlrListStringWithVariables('#modal div[name=expanded-value-provider-value]');
    }
}

global.VariablesPage = VariablesPage;
global.VariableDetails = VariableDetails;
