import ReleaseVariablesPage from '../release-variables-page';
import ReleaseSummaryPage from '../release-summary-page';
import {ReleasePropertiesPage} from '../release-properties-page';
import {
    CreateReleaseTaskDetails,
    CustomScriptDetails,
    ExternalScriptTaskDetails,
    GateDetails,
    NotificationDetails,
    TaskDetails,
    UserInputDetails
} from '../../components/task-details/index';
import ReleaseGridPage from '../release-grid-page';
import ReleasePermissionsPage from './release-permissions';
import ReleaseGanttPage from './release-gantt-page';
import ReleasePhase from './release-phase';
import RestartPhaseModal from '../../components/restart-phase-modal';
import Teams from '../teams-page';
import Page from '../page';
import {ScriptDetails} from "../../components/task-details";
import Download from '../../components/download';
import {TemplatePropertiesPage} from '../properties-page';

export default class ReleasePage {
    constructor(releaseId) {
        this.releaseId = releaseId;
        this.constructorPromise = cy.get('#release-content').should('be.visible');
    }

    open(name) {
        cy.waitForAngular();
        return cy
            .doWithRetry(
                () => cy
                    .get('.subpage-links .dropdown-button')
                    .click(),
                () => expect(Cypress.$(`#subpages-container div:contains('${name}')`)).to.exist)
            .get(`#subpages-container div:contains('${name}')`)
            .click()
    }

    expectTitleToBe(title) {
        cy.get('.bread-crumbs span').should('contain', title);
        return this;
    }

    expectSubPageTitleToBe(title) {
        cy.get('#release-header .subpage-links button').should('contain', title);
        return this;
        // expect(element(By.css('#release-header .subpage-links button')).getText()).toContain(title);
        // return this;
    }

    addPhase() {
        cy.get('.add-phase.button:visible').click();
        return this;
    }

    navigateToPhase(phaseTitle) {
        cy.get('#navigator:visible').should('have.length', 0);
        cy.get('#navigator-toggler').click();
        cy.get('#navigator:visible').should('have.length', 1);
        cy.get(`#navigator .phase-link:contains('${phaseTitle}')`).click();
        cy.get('#navigator-toggler').click();
        cy.get('#navigator:visible').should('have.length', 0);
        return this;
    }

    openVariables() {
        this.open('Variables');
        return new ReleaseVariablesPage();
    }

    openFlow() {
        return this.open('Release flow');
    }

    openEditor() {
        this.openSubPage('Release flow');
        return new ReleasePage();
    }

    openGrid() {
        this.selectView('Table');
        // it renders the grid...
        cy.waitUntilStable(1500, () => Cypress.$('div.ui-grid-row:visible').length);
        return new ReleaseGridPage();
    }

    openGantt() {
        this.selectView('Planner');
        return new ReleaseGanttPage();
    }

    selectView(name) {
        cy.get(`#release-header a:contains('${name}')`).click();
        // return element(By.$(`#release-header a:contains('${name}')`)).click();
    }

    openProperties() {
        this.openSubPage('Properties');
        return new ReleasePropertiesPage();
    }

    openPermissions() {
        this.openSubPage('Permissions');
        return new ReleasePermissionsPage();
    }

    getPhase(phaseName) {
        return new ReleasePhase(`.phase:contains('${phaseName}')`);
    }

    expectPhaseCountToBe(expectedPhaseCount) {
        cy.get('.phase').should('have.length', expectedPhaseCount);
        return this;
    }

    expectVariableToBeFound(variableName) {
        cy.get(`span.variable:contains('${variableName}')`).should('have.length', 1);
    }

    newRelease() {
        cy.get('.new-release').click();
        return new ReleasePropertiesPage();
    }

    downloadExcel() {
        this.exportTemplateAs('Excel');
        Download.expectDownloadAt('api/v1/reports/download');
    }

    expectStarted() {
        this.expectStatus('in_progress');
    }

    start() {
        cy.get('.start-release')
            .should('be.visible')
            .click();
        cy.get(".modal:visible .button:contains('Start')")
            .should('be.visible')
            .click();
        cy.get('.release-status')
            .should('not.contain', 'Pending');
        return this;
    }

    abort(abortComment = "Aborted in testing") {
        cy.get('.abort-release').click();
        cy.get('textarea[ng-model="abortComment"]').type(abortComment, {force: true});
        cy.get('.modal:visible .continue').click();
    }

    expectStatus(status) {
        cy.get(`#release.${status}`).should('have.length', 1);
    }

    openSubPage(name) {
        cy.waitForAngular();
        cy.get('.subpage-links .dropdown-button').click();
        cy.get(`#subpages-container div:contains('${name}')`).click();
        return this;
    }

    exportTemplateAs(name) {
        cy.get('.export-buttons.dropdown-button').click();
        cy.get(`.export-buttons-container div:contains('${name}')`).click();
        return this;
    }

    restartPhases() {
        cy.get('#release-header .restart-phases.button').click();
        return new RestartPhaseModal();
    }

    expectSubPage(name, toBePresent = true) {
        cy.get('#release-header .subpage-links button').click();
        cy.get(`#subpages-container div:contains('${name}')`).should('be.visible');
        Page.clickElseWhere();
        return this;
    }

    expectCurrentPhaseToBe(phaseName) {
        cy.get(`.phase-header.active-phase:contains('${phaseName}')`).should('have.length', 1);
        return this;
    }

    expectNoCurrentPhase() {
        cy.get('.phase-header.active-phase').should('have.length', 0);
        return this;
    }

    openSummary() {
        this.openSubPage('Release dashboard');
        return new ReleaseSummaryPage();
    }

    refresh() {
        Page.softReload();
        return new ReleasePage();
    }

    resume() {
        cy.get('#release-header .resume.button').click();
        return this;
    }

    viewAllCompletedPhases() {
        cy.get('.phase a:contains("View all completed phases")').click();
        return this;
    }

    openTeams() {
        this.openSubPage('Permissions');
        return new Teams();
    }

    exportToZip() {
        this.exportTemplateAs('XLR');
        return Download.expectDownloadAt('export/zip/');
    }

    openTemplateProperties() {
        this.openSubPage('Properties');
        return new TemplatePropertiesPage();
    }

    expectStarted() {
        return this.expectStatus('in_progress');
    }

    expectStatus(status) {
        cy.get(`#release.${status}`).should('have.length', 1);
        return this;
    }
}

const TaskMixin = {

    expectCurrentTasksCountToBe(count) {
        cy.get('.task.active').should('have.length', count);
        return this;
    },
    expectCurrentTaskToContain(title) {
        cy.get(`.task.active .task-title:contains('${title}')`).should('have.length', 1);
        return this;
    },
    expectTaskCompletedInAdvance(title) {
        cy.get(`.task:contains('${title}').completed_in_advance`).should('have.length', 1);
        return this;
    },
    expectTaskInProgress(title) {
        cy.get(`.task:contains('${title}').in_progress`).should('be.visible');
        return this;
    },
    expectTaskPending(title) {
        cy.get(`.task:contains('${title}').pending`).should('be.visible');
        return this;
    },
    expectTaskSkipped(title) {
        cy.get(`.task:contains('${title}') .label:contains('Skipped')`).should('have.length', 1);
        return this;
    },
    expectTaskToHaveIcon(title, icon) {
        cy.get(`.task:contains('${title}') .labels ${icon}`).should('be.visible');
        return this;
    },
    openScriptDetails(title) {
        return this.openTaskDetails(title, () => new ScriptDetails());
    },
    openCustomScriptDetails(title) {
        return this.openTaskDetails(title, () => new CustomScriptDetails());
    },
    openExternalScriptTaskDetails(title) {
        return this.openTaskDetails(title, () => new ExternalScriptTaskDetails());
    },
    openGateTaskDetails(title) {
        return this.openTaskDetails(title, () => new GateDetails());
    },
    openManualTaskDetails(title) {
        return this.openTaskDetails(title, () => new TaskDetails());
    },
    openNotificationDetails(title) {
        return this.openTaskDetails(title, () => new NotificationDetails());
    },
    openCreateReleaseTask(title) {
        return this.openTaskDetails(title, () => new CreateReleaseTaskDetails());
    },
    openTaskDetails(title, buildResult) {
        cy.get(`.task-title:contains('${title}')`).click();
        return buildResult();
    },
    openUserInputDetails(title) {
        return this.openTaskDetails(title, () => new UserInputDetails());
    },
    waitForTaskPlanned(title, timeout = 20) {
        cy.get(`.task:contains('${title}').planned`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskCompleted(title, timeout = 20) {
        cy.get(`.task:contains('${title}').completed`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskCompletedInAdvance(title, timeout = 20) {
        cy.get(`.task:contains('${title}').completed_in_advance`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskFailed(title, timeout = 20) {
        cy.get(`.task:contains('${title}').failed`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskInProgress(title, timeout = 20) {
        cy.get(`.task:contains('${title}').in_progress`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskNeedsInput(title, timeout = 20) {
        cy.get(`.task:contains('${title}').waiting_for_input`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskSkipped(title, timeout = 20) {
        cy.get(`.task:contains('${title}').skipped`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskSkippedInAdvance(title, timeout = 20) {
        cy.get(`.task:contains('${title}').skipped_in_advance`).should('be.visible', {timeout});
        return this;
    },
    waitForTaskStarted(title, timeout = 20) {
        cy.get(`.task:contains('${title}').skipped_in_advance`).should('be.visible', {timeout});
        return this;
    }
};

Object.assign(ReleasePage.prototype, TaskMixin);
