import {TaskDetails} from './task-details';
import {InlineEditor} from '../../pages/inline-editor';
import {XlWidgetAutocomplete} from '../xl-components';
import {VariablesEditor} from '../variables-editor';
import {XlrBooleanWithVariables, XlrDropdownWithVariables} from "../xlr-input-with-variables";
import TagEditor from '../tag-editor';

export class CreateReleaseTaskDetails extends TaskDetails {
    constructor() {
        super();
        this.isVariableMode = false;
    }

    expectVariablesDisplayed(variables, expected = true) {
        this.getVariablesEditor().expectVariablesDisplayed(variables, expected);
        return this;
    }

    setVariableValue(key, value, multiline = false) {
        this.getVariablesEditor().setVariableValue(key, value, multiline);
        return this;
    }

    toggleTemplateId() {
        cy.get('.create-release-details [name="template"] .variable-toggle > button').click();
        this.isVariableMode = !this.isVariableMode;
        return this;
    }

    getTemplateIdEditor() {
        return new XlrDropdownWithVariables('.create-release-details [name="template"]');
    }

    getVariablesEditor() {
        return new VariablesEditor(".create-release-details .variable-editor");
    }

    getReleaseTitleEditor() {
        const editor = new InlineEditor(".create-release-details div[model='createReleaseCtrl.newReleaseTitle']");
        return editor;
    }

    setReleaseTitle(title) {
        const editor = this.getReleaseTitleEditor();
        editor.set(title);
        return this;
    }

    getReleaseTagsEditor() {
        return new TagEditor(".create-release-details div[tags='createReleaseCtrl.releaseTags']");
    }

    addReleaseTags(tags) {
        const editor = this.getReleaseTagsEditor();
        tags.forEach(tag => {
            cy.server();
            cy.route('PUT', '/tasks/**').as('updateTags');
            editor.addTag(tag);
            cy.wait('@updateTags');
        });
        return this;
    }

    expectReleaseTitle(title) {
        const editor = this.getReleaseTitleEditor();
        editor.value().should('eq', title);
        return this;
    }

    getTemplateAutoComplete() {
        return new XlWidgetAutocomplete(".create-release-details div[metadata=\"{name: 'template'}\"]");
    }

    getRiskProfileAutoComplete() {
        return new XlWidgetAutocomplete(".create-release-details div[metadata=\"{name: 'riskProfile'}\"]");
    }

    setTemplate(value, wait = true) {
        cy.server();
        cy.route('PUT', 'tasks/**').as('setTemplate');
        const templateWithValue = this.getTemplateAutoComplete().enter(value);
        cy.wait(1000);
        templateWithValue.select(value);
        if (wait) {
            cy.wait('@setTemplate');
        }
        return this;
    }

    clearTemplate() {
        this.getTemplateAutoComplete().clearByButton();
        return this;
    }

    errorMsgVisible(error) {
        cy.get('.confirm-box .error').should('contain', error);
        return this;
    }

    isBtnVisible(status) {
        cy.get(`span.${status}:visible`).should('be.visible');
        return this;
    }

    clickVisibleButton(status) {
        cy.get(`span.${status}:visible`).click();
        return this;
    }

    confirmTemplateChange(status) {
        cy.get(`.confirm-box span.${status}`).click({force: true});
        return this;
    }

    expectTemplate(value) {
        this.getTemplateAutoComplete().expectValue(value);
        return this;
    }

    setRiskProfile(value) {
        this.getRiskProfileAutoComplete().focus().select(value);
        // // on selenium grid the item doesn't get clicked
        // browser.actions().sendKeys(protractor.Key.ENTER).perform();
        return this;
    }

    clearRiskProfile() {
        this.getRiskProfileAutoComplete().clearByButton();
        return this;
    }

    expectRiskProfile(value) {
        this.getRiskProfileAutoComplete().expectValue(value);
        return this;
    }

    getStartReleaseField() {
        return new XlrBooleanWithVariables(".create-release-details #startRelease");
    }

    setStartReleaseValue(value) {
        this.getStartReleaseField().editor().check(value);
        return this;
    }

    setStartReleaseVariable(variable) {
        this.getStartReleaseField()
            .toggleMode()
            .variableSelector()
            .focus()
            .enter(variable)
            .select(variable);

        return this;
    }

    expectStartReleaseVariable(variable) {
        this.getStartReleaseField()
            .expectVariableMode()
            .variableSelector()
            .expectValue(variable);
        return this;
    }

    getCreatedReleaseIdField() {
        return new XlWidgetAutocomplete(".create-release-details #createdReleaseId");
    }

    setCreatedReleaseIdVariable(variable) {
        this.getCreatedReleaseIdField()
            .scrollModalDown()
            .focus()
            .enter(variable)
            .select(variable);
        return this;
    }

    openCreatedReleaseIdVariable(variable) {
        this.getCreatedReleaseIdField()
            .scrollModalDown()
            .focus();
        return this;
    }

    expectCreatedReleaseIdVariable(variable) {
        this.getCreatedReleaseIdField().expectValue(variable);
        return this;
    }

    expectCreatedReleaseIdValue(value) {
        cy.get(".create-release-details #createdReleaseId").should('contain', value);
    }

    setFolder(folderTitle) {
        cy.get('.Select-input input').click({force:true});
        cy.get('.Select-input input').type(folderTitle, {force: true});
        cy.get(`.xl-react-tree-node:contains('${folderTitle}')`).scrollIntoView().click();
        return this;
    }
}
