import ReleasePage from './release-page';
import DateFilters from '../components/date-filters';
import Page from './page';
import {ReleasePropertiesPage} from './release-properties-page';

// TODO: [REL-5759] change <span> to <a href="javascript:void(0)"> after getting rid of karma e2e
export default class ReleasesListPage {
    constructor() {
        cy.get('#releases').should('be.visible');
        this.waitForReleasesLoaded();
    }

    clickOnFilter(criteria) {
        cy.server();
        cy.route('POST', '/releases/search*').as('clickOnFilter');
        cy.get('.xl-react-widget-tags .xl-react-components-input-full').first().click();
        cy.get('.ci-filter-status .react-tagsinput-input').clear().type(criteria);
        cy.get(`.ci-filter-status .xl-react-autocomplete-suggestion div[title='${criteria}']`).click({force: true});
        cy.wait('@clickOnFilter');
        this.waitForReleasesLoaded();
        return this;
    }

    clickOnFilterButton(criteria) {
        cy.get(`.button-toggle-filter:contains('${criteria}')`).should('be.visible').and('not.have.class', '.active');
        cy.doWithRetry(
            () => cy.get(`.button-toggle-filter:contains('${criteria}')`).click(),
            `.button-toggle-filter.active:contains('${criteria}')`);
        this.waitForReleasesLoaded();
        return this;
    }

    //clicking the search will toggle the filter bar so be careful how many times you click the search
    clickSearch() {
        cy.get('.button-toggle-filter-icon.search-icon').click();
        return this;
    }

    expectBooleanFilters(filters) {
        // cy.get('.release-filter .btn:first-child').click();
        cy.get(`.nav-in-header li:contains('Releases')`).click();
        cy.get('.ci-filter-status').click();
        for (const name in filters) {
            const expectedValue = filters[name];
            cy.get('.ci-filter-status .react-tagsinput-input').click({force: true});
            cy.get('.ci-filter-status .react-tagsinput-input').clear().type(name);
            cy.get(`[title="${name}"] input[type="checkbox"]:checked`).should('have.length', +expectedValue);
            // cy.get(`.release-filter .bootstrap-link:contains('${name}') input[type="checkbox"]:checked`).should('have.length', +expectedValue);
        }
        // cy.get('.filter-title:contains(Status)').click();
        cy.get(`.nav-in-header li:contains('Releases')`).click();
        // Page.clickElseWhere();
        return this;
    }

    expectBulkActionButtonToBeVisible(buttonText) {
        cy.get(`#releases-header button:contains("${buttonText}")`).should('be.visible');
        return this;
    }

    expectBulkActionButtonToBeVisibleInsideFolder(buttonText) {
        cy.get(`#folder-releases-header button:contains("${buttonText}")`).should('be.visible');
        return this;
    }

    expectButtonFilters(filters) {
        for (const name in filters) {
            const expectedValue = filters[name];
            cy.get(`.button-toggle-filter:contains('${name}').active`).should('have.length', +expectedValue);
        }
        return this;
    }

    expectDateFilters(filters, format) {
        DateFilters.expectDateFilters(filters, format);
        return this;
    }

    expectReleaseAborted(title) {
        cy.get(`.release:contains('${title}')`).should('have.length', 0);
        // expect(repeater(`.release:contains('${title}')`).count()).toBe(0);
    }

    expectReleaseByTitleNotToContain(title, text) {
        const $release = cy.get(`.release:contains('${title}')`);
        $release.should('not.contain', text);
        return this;
    }

    expectReleaseByTitleToContain(title, text) {
        cy.wait(800);
        const $release = cy.get(`.release:contains('${title}')`);
        $release.should('contain', text);
        return this;
    }

    expectReleaseDisplayed(release) {
        cy.get(`.release:contains('${release}')`, {timeout: 10000}).should('have.length', 1);
        return this;
    }

    openPipeline(release) {
        cy.get(`.raised-link-wrapper:contains('${release}') .pipeline-button`).click({force: true});
        return this;
    }

    expectPipelineToContainPhase(title) {
        cy.get(`.raised-link-wrapper:contains('${title}') .pipeline .ng-scope div .phase-title`).should('contain', title);
        return this;
    }

    expectReleasesDisplayed(releases) {
        for (const title in releases) {
            const expected = releases[title];
            const expectedCount = 0 + !!expected.displayed;
            if (expected.displayed && expected.index) {
                cy.get(`.release:contains('${title}')`).should('have.length', expectedCount);
                // expect(repeater(`.release:contains('${title}')`).count()).toBe(expectedCount);
                cy.get(`.release:nth('${expected.index}') .release-title`).should('contain', title);
                // expect(element(`.release:nth('${expected.index}') .release-title`).text()).toContain(title);
            }
        }
        return this;
    }

    expectReleaseNotDisplayed(release) {
        cy.get(`.release:contains('${release}')`, {timeout: 10000}).should('have.length', 0);
        return this;
    }

    expectReleasePlanned(title) {
        cy.get(`.release:contains('${title}') .release-status:contains('Planned')`).should('have.length', 1);
        return this;
    }

    expectReleaseArchived(title) {
        cy.get(`.release:contains('${title}') .release-status:contains('Archived')`).should('have.length', 1);
        return this;
    }

    expectReleaseCompleted(title) {
        cy.get(`.release:contains('${title}') .release-status:contains('Completed')`).should('have.length', 1);
        return this;
    }

    expectReleaseStatusAborted(title) {
        cy.get(`.release:contains('${title}') .release-status:contains('Aborted')`).should('have.length', 1);
        return this;
    }

    expectReleaseStarted(title) {
        cy.get(`.release:contains('${title}') .release-status:contains('Planned')`).should('have.length', 0);
        return this;
    }

    expectReleaseFailed(title) {
        cy.get(`.release:contains('${title}') .release-status:contains('Failed')`).should('have.length', 1);
        return this;
    }

    expectReleaseFailing(title) {
        cy.get(`.release:contains('${title}') .release-status:contains('Failing')`).should('have.length', 1);
        return this;
    }

    expectTextFilter(text) {
        cy.get('.searchFilter').should('have.value', text);
        return this;
    }

    refresh() {
        Page.softReload();
        return new ReleasesListPage();
    }

    newRelease() {
        cy.get('.new-release').click();
        return new ReleasePropertiesPage();
    }

    setOrderBy(orderBy) {
        cy.get('.order-by-filter .btn:first-child').click();
        cy.get(`.order-by-filter .bootstrap-link:contains('${orderBy}')`).click();
        cy.get('.order-by-filter .btn:first-child').click();
        this.waitForReleasesLoaded();
        return this;
    }

    resetFilter() {
        this.clearFilters();
        this.clickOnFilter('Planned');
        this.setOrderBy('Risk');
        return this;
    }

    clearFilters() {
        cy.waitForAngular();
        cy.get('.xl-react-widget-tags .xl-react-components-input-full:visible').first().click();
        cy.waitWithoutFailing(`.ci-filter-status .xl-react-link:contains('Clear all'):enabled`, 500)
            .get(`.ci-filter-status .xl-react-link:contains('Clear all')`)
            .then((clearAll) => {
                if (!clearAll.attr('disabled')) {
                    return cy
                        .server()
                        .route('POST', '/releases/search*').as('clearFilters')
                        .get(`.ci-filter-status .xl-react-link:contains('Clear all')`).click({force: true})
                        .wait('@clearFilters');
                }
            });
        return this;
    }

    toggleSearch() {
        cy.get('.toggle-search-filter').click();
        return this;
    }

    showSearch() {
        cy
            .waitUntilStable(200, () => Cypress.$('#title-filter:visible').length)
            .doWithRetry(
                () => cy.get('.toggle-search-filter').click(),
                '#title-filter:visible'
            );
    }

    setSearchFilter(filterText) {
        cy.server();
        cy.route('POST', '/releases/search*').as('setSearchFilter');
        cy.get('.searchFilter').clear().type(filterText).blur();
        cy.wait('@setSearchFilter');
        this.waitForReleasesLoaded();
        return this;
    }

    setSearchFilterNoWait(filterText) {
        cy.get('.searchFilter').type(filterText);
        this.waitForReleasesLoaded();
        return this;
    }

    setTagFilter(filterText) {
        cy.server();
        cy.route('POST', '/releases/search*').as('setTagFilter');
        cy.get('.tag-input').type(filterText);
        cy.get('.tag-input').type(`{enter}`);
        cy.wait('@setTagFilter');
        this.waitForReleasesLoaded();
        return this;
    }

    setAbortComment(abortComment = "Test") {
        cy.get('textarea[ng-model="abortComment"]').type(abortComment);
        return this;
    }

    releaseElement(releaseName) {
        cy.get(`.release:contains('${releaseName}')`).should('be.visible');
        return cy.get(`.release:contains('${releaseName}')`);
    }

    waitForReleasesLoaded() {
        cy.get('.releases-list-loaded:visible').should('be.visible');
        return this;
    }

    waitForReleaseByName(releaseName) {
        cy.get(`.release-line:contains('${releaseName}')`).should('be.visible');
        return this;
    }

    openReleaseByName(releaseName) {
        cy.wait(1000);  // TODO: find a better way to handle this.
        cy.doWithRetry(
            () => cy.get(`.release-line:contains('${releaseName}') .raised-link`).click({force: true}),
            '#release-content');
        return new ReleasePage();
    }

    clickReleaseContextMenu(releaseName) {
        cy.get(`.release-line:contains('${releaseName}') .options-icon`).click({force: true});
        return this;
    }

    clickAddToRelease() {
        cy.get('a:contains("Add to group")').click();
        return this;
    }

    checkAddDisabled() {
        cy.get('.modal-footer .primary').should('be.disabled');
        return this;
    }

    selectReleaseGroup(rgTitle) {
        cy.get('.ui-autocomplete-input').click({force: true});
        cy.get('.ui-autocomplete-input').type(rgTitle, {delay: 100});
        cy.get('.ui-autocomplete-input').type('{backspace}');
        cy.get(`li:contains(${rgTitle})`).click();
        return this;
    }

    clickAddToReleaseGroup() {
        cy.get('.modal-footer .primary').should('be.visible');
        cy.get('.modal-footer .primary').click();
        return this;
    }

    selectReleaseWithIds(releaseIds) {
        releaseIds.forEach(id => {
            cy.get(`.release-row:contains("${id}")`).should('be.visible');
            cy.wait(1200);
            cy.get(`.release-row:contains("${id}") [type="checkbox"]`).click({force: true});
        });
        return this;
    }

    clickOnButton(buttonText) {
        cy.get(`#releases-header button:contains("${buttonText}")`).click();
        return this;
    }

    clickOnButtonInFolder(buttonText) {
        cy.get(`#folder-releases-header button:contains("${buttonText}")`).click();
        return this;
    }

    toggleArchived() {
        cy.get('.only-archived').click();
        return this;
    }

    searchOnlyArchived() {
        cy.get('.only-archived').then((button) => {
            //if is archived selected
            if (button.find('button.button-toggle-filter.active').length > 0) {
                //dont click again
            } else {
                this.toggleArchived();
            }
        });
        return this;
    }

    expectReleaseFlagged(title, flag) {
        cy.get(`.release:contains('${title}').flagged_${flag}`).should('have.length', 1);
    }

    confirmAction() {
        cy.get('.modal-footer .primary').should('be.visible');
        cy.get('.modal-footer .primary').click();
        return this;
    }

    shiftSelectFromTo(from, to) {
        from++;
        to++;

        cy.get(`.release-row:nth-child(${from})`).should('be.visible');
        cy.wait(500);
        cy.get(`.release-row:nth-child(${from}) [type="checkbox"]`).click({force: true});

        cy.get('body').type('{shift}', { release: false });
        cy.get(`.release-row:nth-child(${to}) [type="checkbox"]`).click({force: true});
        cy.get('body').type('{shift}', { release: true });

        return this;
    }

    selectAllReleases() {
        cy.get('#selectAllCheckbox').click();
        return this;
    }

    expectSelectedReleasesLabelToBe(numberOfSelectedReleasesText) {
        cy.get(`#releases-header .unselect-all:contains('${numberOfSelectedReleasesText}')`).should('be.visible');
        return this;
    }

    expectSelectedReleasesLabelToBeVisible(visible = true) {
        cy.get(`#releases-header .filter-release-basic > .unselect-all`).should(visible ? 'be.visible' : 'not.be.visible');
        return this;
    }

    expectNumberOfReleases(title, expectedCount) {
        const releaseSelector = `.release:contains('${title}')`;
        if (expectedCount === 1) {
            cy.get(releaseSelector).should('be.visible');
        } else {
            cy.get(releaseSelector).should('have.length', expectedCount);
        }
        return this;
    }

}
