import _ from 'lodash';

let JIRA_ISSUE_API_URL = "https://xebialabs.atlassian.net/rest/api/2/issue";

let configurationItemsIds = undefined;
let usernames = undefined;
let releaseIds = undefined;
let folderVariableIds = [];
let archivedReleaseIds = undefined;
let globalVariableIds = undefined;
let folderIds = undefined;
let environmentIds = undefined;
let sharedConfigurationIds = [];
let userProfiles = undefined;
let riskProfiles = [];
let reportJobIds = [];
let triggerIds = [];

export const STATUS = {
    IN_PROGRESS: 'in_progress',
    PLANNED: 'planned',
};

export class Fixtures {
    static initClass() {
        configurationItemsIds = [];
        sharedConfigurationIds = [];
        usernames = [];
        releaseIds = [];
        archivedReleaseIds = [];
        folderIds = [];
        userProfiles = [];
        riskProfiles = [];
        environmentIds = [];
        reportJobIds = [];
        triggerIds = [];
    }

    static getParentId(id) {
        if (id.indexOf('/') === -1) return null;
        return id.substring(0, id.lastIndexOf('/'));
    }

    static sendRequest(type, url, data) {
        let retries = 0;
        let recentProblem = '';
        const send = (clearCookies) => {
            if (retries > 5) {
                return; // ignore error
            }
            return (clearCookies ? cy.clearCookies() : cy).request({
                method: type,
                url,
                body: data,
                failOnStatusCode: false,
                headers: {
                    'Authorization': 'Basic YWRtaW46YWRtaW4=',
                    'Cookie': 'XSRF-TOKEN=1;',
                    'X-XSRF-TOKEN': '1'
                },
                timeout: 120000, // our Jenkins can be slow
            }).then((response) => {
                if (! /^2|3/.test(response.status)) {
                    recentProblem = `got this: ${JSON.stringify(response)}`;
                    retries ++;
                    return send(true);
                }
            });
        };
        return send();
    }

    static sendRequestAndIgnoreResult(type, url, data) {
        return Flow.execute(fulfill => Server.doRequest(type, url, data).then(fulfill, fulfill));
    }

    static deleteCI(id) {
        return this.sendRequest('DELETE', `fixtures/configuration/${id}`);
    }

    static deleteArchivedRelease(id) {
        if (id.indexOf('Applications/') === -1) {
            id = `Applications/${id}`;
        }
        return this.sendRequest('DELETE', 'fixtures/cis', [{
            id,
            "type": "xlrelease.Release"
        }]);
    }

    static release(release) {
        initReleaseDefaults(release);
        releaseIds.push(release.id);
        return this.deleteRelease(release.id)
            .then(() => this.deleteArchivedRelease(release.id))
            .then(() => this.sendRequest('POST', 'fixtures/release', release));
    }

    static trigger(trigger) {
        triggerIds.push(trigger.id);
        return this.sendRequest('POST', 'fixtures/trigger', trigger);
    }

    static getAllTriggers() {
        return this.sendRequest('GET', 'api/v1/triggers')
            .then(resp => {
                const searchView = resp.body;
                return searchView.content;
            });
    }

    static preArchivedRelease(release) {
        initReleaseDefaults(release);
        releaseIds.push(release.id);
        archivedReleaseIds.push(release.id);
        return this.deleteRelease(release.id)
            .then(() => this.deleteArchivedRelease(release.id))
            .then(() => this.sendRequest('POST', 'fixtures/preArchived', release));
    }

    static archivedRelease(release) {
        initReleaseDefaults(release);
        archivedReleaseIds.push(release.id);
        return this.deleteRelease(release.id)
            .then(() => this.deleteArchivedRelease(release.id))
            .then(() => this.sendRequest('POST', 'fixtures/archived', release));
    }

    static markReleaseToDelete(id) {
        releaseIds.push(id);
    }

    static markTriggerDeleted(id) {
        triggerIds = _.without(triggerIds, id);
    }

    static ci(ci) {
        configurationItemsIds.push(ci.id);
        return this.sendRequest('POST', 'fixtures', [ci]);
    }

    static configuration(ci) {
        sharedConfigurationIds.push(ci.id);
        return this.sendRequest('POST', 'fixtures/shared', [ci]);
    }

    static cis(cis) {
        for (let ci of cis) {
            configurationItemsIds.push(ci.id);
        }
        return this.sendRequest('POST', 'fixtures', cis);
    }

    static activityLogs(releaseId, logs) {
        return this.sendRequest('POST', `fixtures/activityLogs/${releaseId}`, logs);
    }

    // Release delivery pattern
    static createDeliveryPattern(delivery) {
        return this.sendRequest('POST', 'fixtures/deliveries/patterns', delivery);
    }

    // Release deliveries
    static createDelivery(delivery) {
        return this.sendRequest('POST', 'fixtures/deliveries', delivery);
    }

    static deleteDelivery(delivery) {
        return this.sendRequest('DELETE', `fixtures/deliveries/${delivery}`);
    }

    static deleteRelease(releaseId) {
        if (releaseId.indexOf('Applications/') === -1) {
            releaseId = 'Applications/' + releaseId;
        }
        return this.sendRequest('DELETE', `fixtures/${releaseId}`);
    }

    static addSystemTeams(teams) {
        const systemTeams = ["Folder Owner", "Release Admin", "Template Owner"];
        const teamsToUpdate = teams.map(team => team.teamName);
        systemTeams.forEach((teamName) => {
            if (teamsToUpdate.indexOf(teamName) === -1) {
                teams.push (
                    {
                        type: 'xlrelease.Team',
                        teamName,
                        members: [],
                        permissions: []
                    });
            }
        });
        return teams;
    }

    static folder(folderJson) {
        let parentId = Fixtures.getParentId(folderJson.id);
        folderJson.type = "xlrelease.Folder";

        const folderVariables = (folderJson.variables || []).map(variable => {
            variable.folderId = folderJson.id;
            return variable;
        });
        delete folderJson.variables;

        let teams = folderJson.teams;
        delete folderJson.teams;

        let children = folderJson.children || [];
        delete folderJson.children;

        return this.sendRequest('POST', `api/v1/folders/${parentId}`, folderJson).then((response) => {
            folderIds.push(folderJson.id);
            if (teams) {
                teams = this.addSystemTeams(teams);
                return this.sendRequest('POST', `api/v1/folders/${folderJson.id}/teams`, teams);
            } else {
                return Promise.resolve(response);
            }
        }).then((response) => {
            if (children && children.length > 0) {
                return Promise.all(_.map(children, (child) => {
                    if (response.body.id) {
                        child.id = `${response.body.id}/${child.id}`;
                    }
                    this.folderVariable(folderVariables);
                    this.folder(child);
                }));
            } else {
                this.folderVariable(folderVariables);
                return Promise.resolve(response.body);
            }
        });
    }

    static createFolder(folderJson) {
        folderJson.type = "xlrelease.Folder";
        return this.sendRequest('POST', `fixtures/folders`, [folderJson]);
    }

    static roles(rolePrincipals) {
        return this.sendRequest('POST', 'fixtures/roles/principals', rolePrincipals);
    }

    static deleteRole(roleName) {
        return this.sendRequest('DELETE', `fixtures/role/${roleName}`);
    }

    static permissions(rolePermissions) {
        return this.sendRequest('POST', 'fixtures/roles/permissions/global', rolePermissions);
    }

    static addUser(username, password) {
        usernames.push(username);
        return this.sendRequest('POST', 'fixtures/user', {username, password});
    }

    static deleteUser(username) {
        return this.sendRequest('DELETE', `fixtures/user/${username}`);
    }

    static taskAccesses(taskAccesses) {
        return this.sendRequest('PUT', 'tasks/types-access', taskAccesses);
    }

    static deleteTaskAccesses() {
        return this.sendRequest('DELETE', "fixtures/task-access");
    }

    static addDefaultUser() {
        return this.addUser('Itchy', 'Itchy');
    }

    static addUserProfile(username, profile) {
        if (!profile) {
            profile = {};
        }
        profile.id = username;
        profile.type = 'xlrelease.UserProfile';
        profile.canonicalId = username.toLowerCase();
        userProfiles.push(profile.canonicalId);
        return this.sendRequest('POST', 'fixtures/userProfile', profile);
    }

    static deleteUserProfile(username) {
        return this.sendRequest('DELETE', `fixtures/userProfile/${username}`);
    }

    static updateUserProfile(username, profile) {
        profile.id = username;
        profile.type = 'xlrelease.UserProfile';
        profile.canonicalId = username.toLowerCase();
        if (userProfiles.indexOf(profile.canonicalId) === -1) {
            userProfiles.push(profile.canonicalId);
        }

        return this.sendRequest('PUT', 'fixtures/userProfile', profile);
    }

    static resetDefaultUserProfile() {
        this.updateUserProfile('Itchy', {email: 'itchy@default.com', loginAllowed: true});
    }

    static expectUserProfileExists(username, userEmail, fullName) {
        return this.sendRequest('GET', `fixtures/userProfile?name=${username}&email=${userEmail}&fullName=${fullName}`);
    }

    static expectContainingAttachments(releaseId, tempFile) {
        let ok = function () {
        };
        let failure = function (message) {
            throw `Can't find attachment '${tempFile.getName()}' with content : '${tempFile.getContent()}'. Message from server: '${message}'`;
        };
        if (releaseId.indexOf('Applications/') === -1) {
            releaseId = 'Applications/' + releaseId;
        }
        return this.sendRequest('POST', `fixtures/expectContainingAttachments/${releaseId}`, {name: tempFile.getName(), expectedContent: tempFile.getContent()})
            .then(ok, failure);
    }

    static addJiraTask() {
        let data = {
            fields: {
                project: {key: "RIT"},
                summary: "Test Task",
                description: "Test task description",
                issuetype: {name: "Task"}
            }
        };
        let options = {
            method: `POST`,
            body: data,
            url: JIRA_ISSUE_API_URL,
            auth: {username: 'xlr-jira-testuser@xebialabs.com', password: 'zgsXK6c3oLVkJxrWVJQy9DB7', sendImmediately: true},
            headers: {
                'Origin': 'https://xebialabs.atlassian.net/'
            }
        };
        return cy.request(options).then(resp => {
                return resp.body["key"];
        });
    }

    static expectJiraTaskStatus(taskId, expectedStatus) {
        let options = {
            method: "GET",
            url: `${JIRA_ISSUE_API_URL}/${taskId}`,
            json: true,
            auth: {username: 'xlr-jira-testuser@xebialabs.com', password: 'zgsXK6c3oLVkJxrWVJQy9DB7', sendImmediately: true},
            resolveWithFullResponse: true
        };
        return cy.request(options).then(resp => {
            return resp.body.fields.status.name === expectedStatus;
        });
    }

    static expectJiraTaskSummary(taskId, expectedSummary) {
        return Flow.execute(function (fulfill, reject) {
            let options = {
                method: "GET",
                uri: `${JIRA_ISSUE_API_URL}/${taskId}`,
                json: true,
                auth: {username: 'xlr-jira-testuser@xebialabs.com', password: 'zgsXK6c3oLVkJxrWVJQy9DB7', sendImmediately: true},
                resolveWithFullResponse: true
            };
            return requestPromise(options).then(function (result) {
                let {body} = result;
                let {summary} = body.fields;
                if (summary !== expectedSummary) {
                    reject(`Expected task ${taskId} summary to be '${expectedSummary}' but was '${summary}'. Message from server : ${body}`);
                }
                return fulfill();
            }).catch(result => reject(result.error));
        });
    }


    static globalVariable(variable) {
        variable.id = variable.id || `Configuration/variables/global/Variable_${variable.key.replace('.', '')}`;
        variable.requiresValue = false;
        variable.showOnReleaseStart = false;
        return this.sendRequest('POST', 'api/v1/config/Configuration/variables/global', variable);
    }

    static deleteAllGlobalVariables() {
        return this.sendRequest('DELETE', "fixtures/globalVariables");
    }

    static folderVariable(variableJson) {
        const variables = _.flatten([variableJson]).map(variable => {
            variable.id = null;
            variable.type = variable.type || "xlrelease.StringVariable";
            variable.requiresValue = false;
            variable.showOnReleaseStart = false;
            return variable;
        });
        return this.sendRequest('POST', 'fixtures/folders/variables', variables)
            .then(response => {
                folderVariableIds.concat(response.body.map(variable => variable.id));
                return response;
            });
    }

    static deleteFolderVariable() {
        folderVariableIds.forEach(variableId => this.sendRequest('DELETE', `fixtures/folders/variable/${variableId}`));
        folderVariableIds = [];
    }

    static clearCalendar() {
        return this.sendRequest('DELETE', "fixtures/calendar");
    }

    static addBlackout(name, from, to) {
        return this.sendRequest('POST', 'calendar/blackouts', {
            label: name,
            startDate: from.toDate(),
            endDate: to.toDate(),
            type: "xlrelease.Blackout",
            id: null
        });
    }

    static deleteFolder(folderId) {
        this.sendRequest('DELETE', `api/v1/folders/${folderId}`);
    }

    static deleteSharedConfigurations(configurationIds) {
        return this.sendRequest('DELETE', "fixtures/shared", [configurationIds]);
    }

    static riskProfile(riskProfile) {
        return this.sendRequest('POST', 'fixtures/riskProfiles', riskProfile)
            .then(() => riskProfiles.push(riskProfile.id));
    }

    static deleteRiskProfile(riskProfileId) {
        return this.sendRequest('DELETE', `fixtures/riskProfiles/${riskProfileId}`);
    }

    static addDefaultSmtpServer () {
        const defaultSmtpServer = [{
            type: 'xlrelease.SmtpServer',
            id: 'Configuration/mail/SmtpServer',
            host: 'localhost',
            port: '25',
            fromAddress: 'foo@bar.com',
        }];
        return this.sendRequest('POST', 'fixtures/', defaultSmtpServer);
    }

    static deleteDefaultSmtpServer() {
      return this.sendRequest('DELETE', 'fixtures/configuration/mail/SmtpServer');
    }

    // Release groups
    static createReleaseGroup(releaseGroup) {
        return this.sendRequest('POST', 'fixtures/release-groups', releaseGroup);
    }

    static deleteReleaseGroup(releaseGroup) {
        return this.sendRequest('DELETE', `fixtures/release-groups/${releaseGroup}`);
    }

    static facet(facet) {
        return this.sendRequest('POST', 'fixtures/facets', facet);
    }

    static application(application) {
        return this.sendRequest('POST', 'fixtures/application', application);
    }

    static deleteApplication(applicationId) {
        return this.sendRequest('DELETE', `fixtures/application/${applicationId}`);
    }

    static environment(environment) {
        return this.sendRequest('POST', 'fixtures/environment', environment);
    }

    static deleteEnvironment(environmentId) {
        return this.sendRequest('DELETE', `fixtures/environment/${environmentId}`);
    }

    static reservation(reservation) {
        return this.sendRequest('POST', 'fixtures/reservation', reservation);
    }

    static deleteReservation(id) {
        return this.sendRequest('DELETE', `fixtures/reservation/${id}`);
    }

    static deleteTrigger(triggerId) {
        return this.sendRequest('DELETE', `fixtures/trigger/${triggerId}`);
    }

    static clearTriggers(){
        this.getAllTriggers()
            .each((trigger) => {
                return this.deleteTrigger(trigger.triggerId);
            });
        triggerIds = [];
    }

    static dashboard(dashboard) {
        return this.sendRequest('POST', 'fixtures/dashboard', dashboard);
    }

    static reportJob(reportJob) {
        return this.sendRequest('POST', 'fixtures/reports', reportJob)
            .then(resp => {
                const jobId = resp.body.id;
                if (!_.isNil(jobId)) {
                    reportJobIds.push(jobId);
                }
                return resp.body;
            });
    }

    static deleteReportJob(reportJobId) {
        reportJobIds = _.without(reportJobIds, reportJobId);
        return this.sendRequest('DELETE', `fixtures/reports/${reportJobId}`);
    }

    static clearReportJobs() {
        reportJobIds = [];
        return this.sendRequest('POST', 'fixtures/reports/delete/all', {});
    }

    static clean() {
        for (let username of usernames) {
            this.deleteUser(username);
            this.deleteUserProfile(username);
        }
        for (let triggerId of triggerIds) {
            this.deleteTrigger(triggerId);
        }
        for (let releaseId of releaseIds.reverse()) {
            this.deleteRelease(releaseId);
        }
        for (let folderId of folderIds.reverse()) {
            this.deleteFolder(folderId);
        }
        for (let configurationItemId of configurationItemsIds) {
            this.deleteCI(configurationItemId);
        }
        for (let sharedId of sharedConfigurationIds) {
            this.deleteSharedConfigurations(sharedId);
        }
        for (let userProfile of userProfiles) {
            this.deleteUserProfile(userProfile);
        }
        for (let riskProfile of riskProfiles) {
            this.deleteRiskProfile(riskProfile);
        }
        for (let archiveReleaseId of archivedReleaseIds) {
            this.deleteArchivedRelease(archiveReleaseId);
        }
        this.clearReportJobs();
        this.deleteAllGlobalVariables();
        for (let environmentId of environmentIds) {
            this.deleteEnvironment(environmentId);
        }
        this.deleteFolderVariable();
        releaseIds = [];
        archivedReleaseIds = [];
        folderIds = [];
        configurationItemsIds = [];
        triggerIds = [];
        usernames = [];
        sharedConfigurationIds = [];
        userProfiles = [];
        riskProfiles = [];
        environmentIds = [];
    }
}
Fixtures.initClass();


const initReleaseDefaults = (function () {
    const RELEASE_TYPE = 'xlrelease.Release';
    const PHASE_TYPE = 'xlrelease.Phase';
    const TEAM_TYPE = 'xlrelease.Team';
    const TASK_TYPE = 'xlrelease.Task';
    const COMMENT_TYPE = 'xlrelease.Comment';
    const CONDITION_TYPE = 'xlrelease.GateCondition';
    const DEPENDENCY_TYPE = 'xlrelease.Dependency';
    const LINK_TYPE = 'xlrelease.Link';
    const ATTACHMENT_TYPE = 'xlrelease.Attachment';
    const DASHBOARD_TYPE = 'xlrelease.Dashboard';
    const TRIGGER_TYPE = 'xlrelease.ReleaseTrigger';
    const JIRA_TYPE = 'jira.CreateIssue';
    const DEFAULT_TASK_OWNER = 'Itchy';
    const _ = require('lodash');

    const processTasks = function (task, container, index) {
        if (_.isUndefined(task.type))
            task.type = TASK_TYPE;
        task.id = task.id || `${container.id}/Task${index}`;
        if (_.isUndefined(task.owner) && task.type !== JIRA_TYPE)
            task.owner = DEFAULT_TASK_OWNER;
        if (task.owner === null)
            delete task.owner;

        _.each(task.conditions, function (condition, index) {
            condition.type = CONDITION_TYPE;
            condition.id = `${task.id}/GateCondition${index}`;
        });
        _.each(task.dependencies, function (dependency, index) {
            dependency.type = DEPENDENCY_TYPE;
            dependency.id = `${task.id}/Dependency${index}`;
        });
        _.each(task.links, function (link, index) {
            link.type = LINK_TYPE;
            link.id = `${task.id}/Link${index}`;
        });
        _.each(task.comments, function (comment, index) {
            comment.type = COMMENT_TYPE;
            comment.id = `${task.id}/Comment${index}`;
        });
        _.each(task.tasks, function (subTask, index) {
            processTasks(subTask, task, index);
        });
        _.each(task.templateVariables, function (variable, index) {
            _.defaults(variable, getVariableEntity(variable.value, variable.key, task.id, index));
        });
        _.each(task.attachments, function (attachment, index) {
            attachment.type = ATTACHMENT_TYPE;
            attachment.id = `${task.id}/Attachment${index}`;
        });
        if ('pythonScript' in task) {
            const pythonScript = task.pythonScript;
            pythonScript.id = `${task.id}/PythonScript`;
            pythonScript.customScriptTask = task.id;
        }
    };

    const processPhases = function (phase, release, index) {
        phase.type = PHASE_TYPE;
        phase.id = `${release.id}/Phase${index}`;

        _.forEach(phase.tasks, function (task, index) {
            processTasks(task, phase, index);
        });
    };

    const getVariableEntity = function (value, key, containerId, index, password) {
        const rv = {};
        const keyNoSyntax = key.replace('${', '').replace('}', '');
        rv.id = `${containerId}/Variable${index}`;
        rv.key = keyNoSyntax;
        rv.requiresValue = true;
        rv.showOnReleaseStart = true;
        rv.type = password ? 'xlrelease.PasswordStringVariable' : 'xlrelease.StringVariable';
        if (value) {
            rv.value = value;
        }
        return rv;
    };

    const getValueProviderConfigurationEntity = function (containerId) {
        return {
            id: `${containerId}/valueProvider`,
            variable: containerId
        };
    };

    function getDashboardExtension(dashboard, releaseId) {
        const dashboardExtension = {
            id: `${releaseId}/summary`,
            type: DASHBOARD_TYPE,
            tiles: []
        };
        if (dashboard.tiles) {
            _.forEach(dashboard.tiles, function (tile, index) {
                dashboardExtension.tiles.push(getTileEntity(tile, `${releaseId}/summary`, index));
            });
        }
        return dashboardExtension;
    }

    function getTileEntity(tile, containerId, index) {
        tile.id = tile.id || `${containerId}/Tile${index}`;
        return tile;
    }

    return function (release) {
        release.type = RELEASE_TYPE;
        if (release.id.indexOf('Applications/') === -1) {
            release.id = `Applications/${release.id}`;
        }
        if (release.startDate) {
            release.queryableStartDate = release.startDate;
        } else if (release.scheduledStartDate) {
            release.queryableStartDate = release.scheduledStartDate;
        }
        if (release.endDate) {
            release.queryableEndDate = release.endDate;
        } else if (release.dueDate) {
            release.queryableEndDate = release.dueDate;
        }
        if (_.isUndefined(release.owner))
            release.owner = 'Itchy'; // default release manager

        _.forEach(release.phases, function (phase, index) {
            processPhases(phase, release, index);
        });
        _.forEach(release.teams, function (team, index) {
            team.type = TEAM_TYPE;
            team.id = `${release.id}/Team${index}`;
        });
        _.forEach(release.attachments, function (attachment, index) {
            attachment.type = ATTACHMENT_TYPE;
            attachment.id = `${release.id}/Attachment${index}`;
        });
        _.forEach(release.variables, function (variable, index) {
            _.defaults(variable, getVariableEntity(variable.value, variable.key, release.id, index));
            if (variable.valueProvider) {
                _.defaults(variable.valueProvider, getValueProviderConfigurationEntity(variable.id));
            }
        });
        _.forEach(_.toPairs(release.variableValues), function (keyValue, index) {
            if (!release.variables)
                release.variables = [];
            release.variables.push(getVariableEntity(keyValue[1], keyValue[0], release.id, 1000 + index));
            release.variableValues = undefined;
        });
        _.forEach(_.toPairs(release.passwordVariableValues), function (keyValue, index) {
            if (!release.variables)
                release.variables = [];
            release.variables.push(getVariableEntity(keyValue[1], keyValue[0], release.id, 1500 + index, true));
            release.passwordVariableValues = undefined;
        });

        if (release.summary) {
            release.extensions = [ getDashboardExtension(release.summary, release.id) ];
            release.summary = undefined;
        }
    };
})();
