export class XlWidgetAutocomplete {
    constructor(selector) {
        this.widget = `${selector}`;
        this.input = `${selector} input:visible`;
    }

    focus() {
        cy.get(this.input).click();
        return this;
    }

    blur() {
        cy.get(this.input).focus().blur({force: true});
        return this;
    }

    scrollModalDown() {
        cy.get('.modal-body').scrollTo('bottom');
        return this;
    }

    getInputValue() {
        return cy.get(this.widget + ' div:nth(0)').invoke('text');
    }

    expectOptions(options) {
        cy.get('ul.xl-components.xl-widget-autocomplete li').should(($lis) => {
            expect($lis).to.have.length(options.length);
            options.map((item, i) =>
                expect($lis.eq(i)).to.contain(options[i])
            );
        });
        return this;
    }

    expectMinimumOptions(options) {
        this.getOptions().get("li").invoke('text').then(function (itemsList) {
            for (const option in options) {
                expect(itemsList).to.contain(options[option]);
            }
        });
        return this;
    }

    expectValue(value) {
        cy.get(this.input).should('have.value', value);
        return this;
    }

    expectNoValue() {
        cy.get(this.input).should('have.value', '');
        return this;
    }

    enter(text) {
        cy.get(this.widget).scrollIntoView();
        cy.get(this.input).clear().type(text, {force: true});
        return this;
    }

    select(name) {
        cy.get(`ul.xl-components.xl-widget-autocomplete:not([style*="display"]) li:contains('${name}')`)
            .first()
            .trigger('mouseover', {force: true})
            .click({force: true});
        return this;
    }

    getOptions() {
        if (this.options) {
            return this.options;
        } else {
            this.focus();
            return this.options = cy.get("ul.xl-components.xl-widget-autocomplete");
        }
    }

    clearByKeys() {
        cy.get(this.input).clear();
        return this;
    }

    clearByButton() {
        cy.get(this.widget).scrollIntoView();
        cy.get(`${this.widget} .xl-btn-close:visible`).click();
        return this;
    }
}

export class XlDipListOfCi extends XlWidgetAutocomplete {
    constructor(selector) {
        super(selector);
        this.selectedList = cy.get(`${selector} ul[ui-sortable]`);
    }

    expectSelected(selected) {
        this.selectedList.get('.xl-scrollable-panel').then(el => {
            if (el.children().is('li')) {
                expect(el.children).to.have.length(selected.length);
                expect(el.children).should('contain', selected[i]);
            }
        });
        return this;
    }

    remove(name) {
        this.selectedList.get(`li:contains('${name}') .xl-btn-close`).click();
        return this;
    }
}

class XlDipInteger {
    constructor(selector) {
        this.input = cy.get(`${selector} .editor input`);
    }

    clear() {
        this.input.clear();
        return this;
    }

    blur() {
        Browser.clickElseWhere();
        return this;
    }

    focus() {
        this.input.click();
        return this;
    }

    enter(text) {
        this.input.sendKeys(text);
        return this;
    }

    set(text) {
        this.focus();
        this.clear();
        this.enter(text);
        return this;
    }
}

export class XlDipCheckbox {
    constructor(selector) {
        this.input = cy.get(`${selector} input[type=checkbox]`);
    }

    check(value = true) {
        const target = this.input;
        return target.isSelected().then(function (selected) {
            if (selected !== value) {
                return target.click();
            }
        });
    }
}
