// ***********************************************
// This example commands.js shows you how to
// create the custom command: 'login'.
//
// The commands.js file is a great place to
// modify existing commands and create custom
// commands for use throughout your tests.
//
// You can read more about custom commands here:
// https://on.cypress.io/api/commands
// ***********************************************
/* eslint-disable angular/json-functions */
import 'cypress-pipe';

import Errors from './dsl/components/errors';

const moment = Cypress.moment;
const RUN_PID_FILE = './screenDiff.json';

function _readRunIdFile() {
    return cy.readFile(RUN_PID_FILE);
}

Cypress.Commands.add("login", (username, password, useAngularLocation = false, clearCookiesAndStorage = false) => {
    const login = username || "admin";
    const pass = password || "admin";

    const log = Cypress.log({
        name: "login",
        message: [login, pass],
        consoleProps: () => (
            login,
                pass
        )
    });

    if (useAngularLocation) {
        cy.navigate('/login');
    } else {
        cy.deepClean().visit("#/login", {timeout: 180000}); // Jenkins can be slow
    } // check if previous tests open the conditional modal for with saving state
    cy.get('body').then(($body) => {
        // synchronously query from body
        // to find which element was created
        if ($body.find(".button:contains('Leave')").length) {
        // modal was found so closing it
            cy.get(".button:contains('Leave')").click();
        }
    })
        .get("#inputLogin").type(login)
        .get("#inputPassword").type(pass)
        .then(() => {
            if (clearCookiesAndStorage) {
                return cy.clearCookies().clearLocalStorage();
            }
        })
        .get(".login-button button").click()
        .get('#releases-content').should('be.visible')
        .location('hash').should('not.eq', '#/login')
        .wait(1000)
        .waitWithoutFailing([('div.release-line:visible'), ('div.alert.notice.releases-list-loaded:visible:contains(No releases found.)')], 2000)
        .then(() => {
            log.snapshot().end();
        });
});
// todo fix
Cypress.Commands.add("logout", () => {
    cy.get('#logout').click({force: true});
    cy.waitWithoutFailing('.login-dialog');
    cy.get('.login-dialog').should('exist');
});

Cypress.Commands.add("loginWithoutReload", (username, password) => {
    cy.get("#inputLogin").type(username);
    cy.get("#inputPassword").type(password);
    cy.get(".login-button button").click();
});

Cypress.Commands.add('expectLocalDateValue', (selector, date, format) => {
    cy.get(selector)
        .then($el => cy.window().its('angular')
            .then(ng => ng.element($el).injector()))
        .then(injector => {
                const dateFilter = injector.get('$filter')('date');
                const formattedDate = dateFilter(date, format);
                cy.get(selector).should('have.value', formattedDate);
            }
        );
});

Cypress.Commands.add('expectLocalDateText', (selector, date, format) => {
    cy.get(selector)
        .then($el => cy.window().its('angular')
            .then(ng => ng.element($el).injector()))
        .then(injector => {
                const dateFilter = injector.get('$filter')('date');
                const formattedDate = dateFilter(date, format);
                cy.get(selector).invoke('text').should('eq', formattedDate);
            }
        );
});

Cypress.Commands.add("compareToLocalDate", (timeStamp, locator, options) => {

    const epochTimeStamp = moment(timeStamp, 'DD-MM-YYYY').utc();
    const date = new Date(epochTimeStamp);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        cy.get(locator).should('have.value', date.toLocaleDateString(locale, options));
    });

});

Cypress.Commands.add("containLocalDate", (timeStamp, locator, options) => {

    const epochTimeStamp = moment(timeStamp, 'DD-MM-YYYY').utc();
    const date = new Date(epochTimeStamp);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        cy.get(locator).should('contain', date.toLocaleDateString(locale, options));
    });

});

Cypress.Commands.add("containLocalTime", (timeStamp, locator, options) => {

    const timeToCompare = moment(timeStamp, "hh:mm");
    const date = new Date(timeToCompare);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        const formatedTime = date.toLocaleTimeString(locale, options);
        cy.get(locator).should('contain', formatedTime);
    });

});

Cypress.Commands.add("compareToLocalTime", (timeStamp, locator, options) => {

    const timeToCompare = moment(timeStamp, "hh:mm");
    const date = new Date(timeToCompare);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        const formatedTime = date.toLocaleTimeString(locale, options);
        cy.get(locator).should('have.value', formatedTime);
    });

});

const diffPercentage = 0.2; // minimum allowed screen diff perentage of visual review server

Cypress.Commands.add("form_request", (method, url, formData) => {
    return cy
        .server()
        .route(method, url).as('postImage')
        .window()
        .then((win) => {
            const xhr = new win.XMLHttpRequest();
            xhr.open(method, url);
            xhr.send(formData);
        })
        .wait('@postImage')
        .its('response')
        .then((resp) => {
            let status = true;
            cy.readFile(RUN_PID_FILE).then((run) => {
                const url = `${Cypress.env('vrServer')}/api/runs/${run.body.id}/analysis`;
                cy.request(url).then((analysis) => {
                    for (const item of analysis.body.diffs) {
                        if (item.after.imageId === resp.body.imageId) {
                            if (item.percentage >= diffPercentage && !Cypress.env("doBaseLine")) {
                                status = false;
                            }
                            expect(status, 'image comparison successful ?').to.be.true;
                        }
                    }
                });
            });
        });
});

Cypress.Commands.add("takeScreenShot", (fileName, options) => {

    if (Cypress.env("disableScreenDiff")) {
        return true;
    }
    cy.wait(1000);
    // take screenshot
    cy.screenshot(fileName, {
        blackout: (options && options.blackout) ? options.blackout : [],
        capture: (options && options.capture) ? options.capture : 'fullPage',
        clip: (options && options.clip) ? options.clip : null,
        disableTimersAndAnimations: (options && options.disableTimersAndAnimations) ? options.disableTimersAndAnimations : false,
        scale: (options && options.scale) ? options.scale : false,
        onBeforeScreenshot: (options && options.onBeforeScreenshot) ? options.onBeforeScreenshot : null,
        onAfterScreenshot($el, props) {
            Cypress.env('props', props);
        },
    }).then(() => {
        // get current run information
        _readRunIdFile().then((run) => {
            const formData = new FormData();
            const method = 'POST';
            const url = `${Cypress.env('vrServer')}/api/runs/${run.body.id}/screenshots`;
            const fileType = 'image/png';
            const path = Cypress.env('props').path;
            const newPath = path.split(Cypress.config('screenshotsFolder'))[1];
            // get the file and send it to visual review server for image diff
            cy.fixture(`../../../../../xlr-ui/build/cypress/screenshots/${newPath}`, 'base64').then((img) => {
                cy.task('log', `${Cypress.env('props').name} is generated`);
                Cypress.Blob.base64StringToBlob(img, fileType).then((blob) => {
                    formData.set('file', blob, `${Cypress.env('props').name}.png`);
                    formData.set('screenshotName', Cypress.env('props').name);
                    formData.set('properties', JSON.stringify({
                        resolution: Cypress.env('props').dimensions.width + 'x' + Cypress.env('props').dimensions.height,
                        browser: 'chrome'
                    }));
                    formData.set('compareSettings', JSON.stringify({
                        "precision": diffPercentage
                    }));
                    formData.set('meta', JSON.stringify({}));
                    // make call and compare images
                    cy.form_request(method, url, formData);
                });
            });
        });
    });
});

Cypress.Commands.add("cleanErrors", () => {
    Errors.clearErrors();
});

const getAngular = () =>
    cy.window()
        .its('angular');


const loadApplicationIfNotLoaded = (url, closure) => {
    cy.hash().then(path => {
        if (path === '' || !path.startsWith('#/')) {
            // application is not loaded
            cy.visit(`#${url.startsWith('/') ? url : '/' + url}`);
        } else {
            closure();
        }
    });
};

Cypress.Commands.add("navigate", (url) => {
    loadApplicationIfNotLoaded(url, () => {
        cy.get('body')
            .then($el =>
                getAngular()
                    .then(angular => {
                            const $injector = angular.element($el).injector();
                            const $rootScope = $injector.get('$rootScope');
                            const $location = $injector.get('$location');
                            $location.url(url);
                            $rootScope.$digest();
                        }
                    )
            );
    });
});

Cypress.Commands.add('stateReload', () => {
    loadApplicationIfNotLoaded('#/login', () => {
        cy.get('body')
            .then($el =>
                getAngular()
                    .then(angular => {
                            const $injector = angular.element($el).injector();
                            const $rootScope = $injector.get('$rootScope');
                            const $stateParams = $injector.get('$stateParams');
                            const $state = $injector.get('$state');
                            $state.transitionTo($state.current, $stateParams, {
                                reload: true, inherit: false, notify: true, location: false
                            });
                            $rootScope.$digest();
                        }
                    )
            );
    });
});

Cypress.Commands.add('waitForPopover', (buttonSelector) => {
    cy.doWithRetry(
        () => cy.get(buttonSelector).click(),
        () => {
            expect(Cypress.$(buttonSelector).next()).to.have.class('popover');
            expect(Cypress.$('.popover-content')).to.be.visible;
        });
    cy.get('.popover-content').should('be.visible');
});

Cypress.Commands.add('doWithRetry', (action, assertions, recover = null, retryTimeout = 5000) => {
    if (!(assertions instanceof Array)) {
        assertions = [assertions];
    }
    assertions = assertions.map(wrapCondition);
    let retries = 0;
    let retryUntil = new Date().getTime() + 30000;
    let i;
    const attempt = () => {
        let actionResult;
        try {
            actionResult = (retries && recover) ? recover() : action();
        } catch (e) {
            return Promise.reject(e);
        }
        const next = () => {
            let waitUntil = new Date().getTime() + retryTimeout;
            return new Promise((res, rej) => {
                function f() {
                    let exception;
                    for (i = 0; i < assertions.length; i ++) {
                        try {
                            assertions[i]();
                            res();
                            return;
                        } catch (e) {
                            exception = e;
                        }
                    }
                    if (new Date().getTime() < waitUntil) {
                        setTimeout(f, 150);
                    } else {
                        retries ++;
                        if (new Date().getTime() < retryUntil) {
                            cy.log('doWithRetry', `retry #${retries}`);
                            res(attempt());
                        } else {
                            rej(exception);
                        }
                    }
                }
                f();
            });
        };
        if (actionResult && 'function' === typeof actionResult.then) {
            return actionResult.then(next);
        } else {
            return Promise.resolve().then(next);
        }
    };
    return cy.wrap().then(() => attempt());
});

Cypress.Commands.add('clickWithRetry', { prevSubject: true }, ($el, assertions) => {
    return cy.doWithRetry(
        () => $el.click(),
        assertions
    );
});

const wrapCondition = (condition) => {
    if ('function' === typeof condition) {
        return condition;
    } else if ('string' === typeof condition) {
        return () => expect(Cypress.$(condition)).to.exist;
    } else {
        throw new Error('Condition should be either a function or a selector string');
    }
};

Cypress.Commands.add('deepClean', () => {
    cy
        .waitUntilStable(300, () => cy.window().then(w => w.location.href))
        .window().then(w => {
            try {
                w.onbeforeunload = null;
            } catch (e) {}
        })
        .visit('/static/asdf/partials/login/login.html', {timeout: 180000})
        .window().then(w => {
            w.sessionStorage.clear();
            w.localStorage.clear();
        })
        .clearCookies().clearLocalStorage();
});

/**
 * The idea of this function is to give some grace period for page to stabilize, not too big (not 30 seconds).
 * Example usages are - we are waiting for tiles to appear within 2 seconds, but if they don't
 * that's also ok, we will consider, that there are no tiles to display. Thus, if tiles will appear
 * earlier - then we'll not wait for 2 seconds and will proceed faster.
 */
Cypress.Commands.add('waitWithoutFailing', (conditions, timeout = 5000) => {
    cy.log('waitWithoutFailing', conditions);
    if (!(conditions instanceof Array)) {
        conditions = [conditions];
    }
    conditions = conditions.map(wrapCondition)
    return cy.wrap().then(() => {
        let tries = 300;
        let when = timeout ? (new Date().getTime() + timeout) : 0;
        return new Promise((res, rej) => {
            function f() {
                let e;
                if (when && new Date().getTime() > when) {
                    res();
                    return;
                }
                let i;
                for (i = 0; i < conditions.length; i ++) {
                    try {
                        conditions[i]();
                        res();
                        return;
                    } catch (e) {
                    }
                }
                if (tries --) {
                    setTimeout(f, 150);
                } else {
                    rej(e);
                }
            }
            f();
        });
    });
});

Cypress.Commands.add('waitUntilStable', (timeout, conditions) => {
    if (!(conditions instanceof Array)) {
        conditions = [conditions];
    }
    const values = conditions.map(v => null);
    let seenChange = new Date().getTime();
    let i = 0;
    const checkCondition = () => {
        return cy
            // .log(`waitUntilStable[${i}, ${new Date().getTime()}]: ${new Date().getTime() - seenChange} vs ${timeout}`)
            .then(() => conditions[i]())
            .then((value) => {
                let outcome;
                if (values[i] !== value) {
                    seenChange = new Date().getTime();
                    values[i] = value;
                    outcome = `waitUntilStable[${i}, ${new Date().getTime()}]: changed from ${values[i]} to ${value}`;
                } else {
                    outcome = `waitUntilStable[${i}, ${new Date().getTime()}]: same ${value}`;
                }
                i++;
                if (i === conditions.length) {
                    if (seenChange >= (new Date().getTime() - timeout)) {
                        i = 0;
                        return cy
                            // .log(outcome)
                            .then(() => new Promise(res => setTimeout(res, 50)))
                            .then(checkCondition);
                    } else {
                        return cy
                            // .log(`${outcome}: completed`)
                            .then(() => {});
                    }
                } else {
                    return cy
                        // .log(outcome)
                        .then(checkCondition);
                }
            });
    };
    return checkCondition();
});
