import DateTimePicker from '../date-time-picker';
import {CodeInlineEditor, InlineEditor, TextareaInlineEditor, UserInlineEditor} from '../../pages/inline-editor';
import {FacetSection} from './facet-section';

const moment = Cypress.moment;

export class TaskDetails {

    constructor() {
        cy.get('#resizable-modal');
    }

    assignTo(name) {
        const editor = new UserInlineEditor('#resizable-modal .task-owner');
        editor.focus();
        editor.enter(name);
        editor.blur();
        return this;
    }

    setVariable(variable, value) {
        const editor = new InlineEditor(`#resizable-modal .form-group:contains('${variable}') div[inline-text-editor]`);
        editor.set(value);
        return this;
    }

    assignUsingCompletionTo(name) {
        const editor = new UserInlineEditor('#resizable-modal .task-owner');
        editor.set(name);
        return this;
    }

    assignToMe() {
        cy.get('#resizable-modal .assign-to-me').click();
        return this;
    }

    unAssign() {
        cy.get(".modal:visible .remove-owner").click();
        return this;
    }

    removeTeam() {
        cy.get(".modal:visible .remove-team").click();
        return this;
    }

    setTitle(title) {
        const editor = new InlineEditor('.modal:visible .task-title');
        editor.set(title);
        return this;
    }

    setDescription(description) {
        const editor = new TextareaInlineEditor('.modal:visible .task-description');
        editor.set(description);
        return this;
    }

    getDueDateTimePicker() {
        return new DateTimePicker('#resizable-modal .due-date');
    }

    close() {
        cy.get('.modal button.close').click();
        return this;
    }

    closePromise() {
        return cy.get('.modal button.close').click();
    }

    getDescriptionEditor() {
        return new TextareaInlineEditor("#resizable-modal .task-description");
    }

    getScheduledStartDateTimePicker() {
        return new DateTimePicker('#resizable-modal .scheduled-start-date');
    }

    getStartDateTimePicker() {
        return new DateTimePicker('#resizable-modal .start-date');
    }

    setDueDateDay(day) {
        this.getDueDateTimePicker().setDay(day);
        return this;
    }

    setDueTime(time) {
        this.getDueDateTimePicker().setTime(time);
        return this;
    }

    checkDelayDuringBlackoutPeriod() {
        cy.get("#delay-during-blackout:checked").should('have.length', 0);
        cy.get("#delay-during-blackout").click();
        cy.get("#delay-during-blackout:checked").should('have.length', 1);
        return this;
    }

    checkEnvironment() {
        cy.get("#check-environment:checked").should('have.length', 0);
        cy.get("#check-environment").click();
        cy.get("#check-environment:checked").should('have.length', 1);
        return this;
    }

    startTask() {
        cy.get("#resizable-modal .button:contains('Start'):visible").click();
        return this;
    }

    startNow(comment, withWarningText) {
        cy.get(".button:contains('Start now'):visible").click();
        cy.get("textarea[ng-model=feedback]:visible").should('have.length', 1);
        if (withWarningText) {
            cy.get(`.task-actions-info span.error:contains('${withWarningText}')`).should('be.visible');
        }
        cy.get(".button:contains('Start now'):visible").should('have.length', 0);
        cy.get("#resizable-modal textarea[ng-model=feedback]:visible").type(comment);
        cy.get(".button:contains('OK')").click();
        return this;
    }

    failTask(comment) {
        cy.get(".button:contains('Fail'):visible").click();
        cy.get("#resizable-modal textarea[ng-model=failFeedback]:visible").should('be.visible');
        cy.get("#resizable-modal textarea[ng-model=failFeedback]:visible").type(comment);
        cy.get("form[name=failTaskForm] button:contains('OK')").click();
        return this;
    }

    retryTask(comment) {
        cy.get(".button:contains('Retry'):visible").click();
        cy.get("#resizable-modal textarea[ng-model=retryFeedback]:visible").should('be.visible');
        cy.get("#resizable-modal textarea[ng-model=retryFeedback]:visible").type(comment);
        cy.get("form[name=retryTaskForm] button:contains('OK')").click();
        return this;
    }

    completeTask() {
        cy.get(".button:contains('Complete'):visible").click();
        cy.get("#resizable-modal textarea[ng-model=feedback]:visible").should('be.visible');
        cy.get("form[name=completeTaskForm] button:contains('OK')").click();
        return this;
    }

    completeTaskPromise() {
        cy.get(".button:contains('Complete'):visible").click();
        cy.get("#resizable-modal textarea[ng-model=feedback]:visible").should('be.visible');
        return cy.get("form[name=completeTaskForm] button:contains('OK')").click();
    }

    skipTask(comment) {
        cy.get(".button:contains('Skip'):visible").click({force: true});
        cy.get("#resizable-modal textarea[ng-model=skipFeedback]:visible").should('be.visible');
        cy.get("#resizable-modal textarea[ng-model=skipFeedback]:visible").type(comment);
        cy.get("form[name=skipTaskForm] button:contains('OK')").click();
        cy.get("#resizable-modal textarea[ng-model=skipFeedback]:visible").should('have.length', 0);
        return this;
    }

    reopenTask(comment) {
        cy.get(".button:contains('Reopen'):visible").click();
        cy.get("#resizable-modal textarea[ng-model=reopenFeedback]:visible").should('be.visible');
        cy.get("#resizable-modal textarea[ng-model=reopenFeedback]:visible").type(comment);
        cy.get("form[name=reopenTaskForm] button:contains('OK')").click();
        return this;
    }

    startNowToBeDisabledDueToBlackout() {
        cy.get(".button:contains('Start now'):visible").should('have.attr', 'disabled', 'disabled');
        cy.get(".task-actions span.error").should('contain',
            "You can't start this task now because it's subject to the current blackout period."
        );
        return this;
    }

    flag(flagName, flagComment) {
        cy.get('#resizable-modal .flags .dropdown-button').click();
        cy.get('#resizable-modal .flags .popover').should('have.length', 1);
        cy.get(`#resizable-modal .flags .flag-picker-popover .${flagName}`).click();
        const editor = new InlineEditor('#resizable-modal .flag-comment');
        editor.set(flagComment);
        return this;
    }

    expectDelayDuringBlackoutToBeDisabled() {
        cy.get('#delay-during-blackout:checked').should('have.attr', 'disabled', 'disabled');
        return this;
    }

    expectTaskIsCompletable(condition) {
        cy.get(".button:contains('Complete'):visible").should('be.visible');
        cy.get(".button:contains('Complete'):enabled").should(`${condition ? '' : 'not.'}be.visible`);
        return this;
    }

    expectScheduledStartDateToBe(formattedDate) {
        cy.get('.modal .scheduled-start-date .date .day').should('contain', formattedDate.format('DD'));
        cy.get('.modal .scheduled-start-date .date .date-detail').should('contain', formattedDate.format('MMMM'));
        cy.get('.modal .scheduled-start-date .date .date-detail').should('contain', formattedDate.format('YYYY'));
        return this;
    }

    expectCompleteButtonEnabled() {
        cy.get("#resizable-modal .button:contains('Complete'):enabled").should('have.length', 1);
        return this;
    }

    expectStartDateToBeDisplayed() {
        cy.get('.modal .start-date').should('have.length', 1);
        return this;
    }

    expectEndDateToBeDisplayed() {
        cy.get('.modal .end-date').should('have.length', 1);
        return this;
    }

    expectBlackoutAlertInProgress() {
        cy.get('.blackout-info:contains("Start date set due to blackout")').should('be.visible');
        return this;
    }

    expectBlackoutOriginalScheduledStartDate(time) {
        cy.get('.original-date').then(date => {
            const d = date.text();
            expect(moment(new Date(d)).unix() === moment(time).unix()).to.be.true;
        });
        return this;
    }

    expectBlackoutAlertCompleted() {
        cy.get('.blackout-info:contains("Task postponed due to blackout")').should('be.visible');
        return this;
    }

    expectBlackoutAlertToNotBePresent() {
        cy.get('.blackout-info').should('not.be.visible');
        return this;
    }

    expectTitleToContain(title) {
        cy.get('#resizable-modal .modal-header').should('contain', title);
        return this;
    }

    expectDescriptionToContain(description) {
        cy.get('.modal:visible .task-description').should('contain', description);
        return this;
    }

    expectDescriptionToContainMarkdown(description) {
        cy.get('.modal:visible .task-description:nth(0)').scrollIntoView();
        cy.get(`.modal:visible .task-description h1:contains('${description}'):visible`).should('have.length', 1);
        return this;
    }

    expectVariableInTitle(variable) {
        cy.get(`#resizable-modal .modal-header span.variable:contains('${variable}'):visible`).should('have.length', 1);
        return this;
    }

    expectCommentsToContain(text) {
        cy.get("#resizable-modal .comments").should('contain', text);
        return this;
    }

    expectCommentsAuthorToBe(author) {
        cy.get("#resizable-modal .comments").should('contain', author);
        return this;
    }

    expectCommentsToContainMarkdown(description) {
        cy.get(`.modal:visible .comment h1:contains('${description}'):visible`).should('have.length', 1);
        return this;
    }

    expectOneCommentDateToBe(date, format) {
        return cy.containLocalDate(date, '.pull-right.comment-date', format);
    }

    expectOneCommentTimeToBe(time, format) {
        return cy.containLocalTime(time, '.pull-right.comment-date', format);
    }

    expectAssigneeToBeEnabled() {
        cy.get('.modal:visible .assignee').should('have.length', 1);
        return this;
    }

    expectAssignedTo(username) {
        cy.get('#resizable-modal .task-owner').should('contain', username);
        return this;
    }

    expectNotAssignedTo(username) {
        cy.get(`.modal:visible .task-owner:visible:contains('${username}')`).should('have.length', 0);
        return this;
    }

    expectCompleteButtonHidden() {
        return this.expectButtonNotVisible('Complete');
    }

    expectSkipButtonEnabled() {
        return this.expectButtonEnabled('Skip');
    }

    expectButtonVisible(buttonName) {
        cy.get(`#resizable-modal .button:contains('${buttonName}'):visible`).should('have.length', 1);
        return this;
    }

    expectButtonNotVisible(buttonName) {
        cy.get(`#resizable-modal .button:contains('${buttonName}'):visible`).should('have.length', 0);
        return this;
    }

    expectButtonEnabled(buttonName) {
        cy.get(`#resizable-modal .button:contains('${buttonName}'):enabled`).should('have.length', 1);
        return this;
    }

    expectButtonDisabled(buttonName) {
        cy.get(`#resizable-modal .button:contains('${buttonName}'):disabled`).should('have.length', 1);
        return this;
    }

    expectModalClosed() {
        cy.get(".modal:hidden").should('have.length', 2);
        return this;
    }

    toggleComments() {
        // cy.get('.comments .inline-label').click();
        cy.get('.comments .inline-label').click();
        return this;
    }

    addComment(comment = "myUsefulComment") {
        cy.get('#resizable-modal .add-comment').click();
        cy.get('.addCommentForm .new-comment').type(comment);
        cy.get("#resizable-modal .save-comment").click();
        return this;
    }

    expectScheduledStartDateToBeDisplayed() {
        cy.get('#resizable-modal .scheduled-start-date').should('have.length', 1);
        return this;
    }

    expectDueDateToBeDisplayed() {
        cy.get('#resizable-modal .due-date').should('have.length', 1);
        return this;
    }

    expectScheduledStartDateDayToBe(day) {
        this.getScheduledStartDateTimePicker().expectDateToContain(day);
        return this;
    }

    uncheckWaitForScheduledDate() {
        this.expectWaitForScheduledDateChecked();
        cy.get('#wait-for-scheduled-start-date').click();
        cy.get('#wait-for-scheduled-start-date:checked').should('have.length', 0);
        return this;
    }

    expectWaitForScheduledDateChecked() {
        cy.get('#wait-for-scheduled-start-date:checked').should('have.length', 1);
        return this;
    }

    expectTaskIsStartable(condition) {
        cy.get(".button:contains('Start'):enabled").should(`${condition ? '' : 'not.'}be.visible`);
        return this;
    }

    expectTaskIsEditable(isEditable) {
        cy.get("#resizable-modal .task-description[xlr-inline-textarea-with-variables]")
            .should(`${isEditable ? '' : 'not.'}be.visible`);
        return this;
    }

    expectWithComment(comment) {
        cy.get('.comments span').first().click();
        cy.get('#resizable-modal .comment').should('contain', comment);
        return this;
    }

    expectAbortButtonHidden() {
        cy.get('#resizable-modal .button:contains("Abort"):visible').should('have.length', 0);
        return this;
    }

    expectFlaggedWith(flagName, flagComment) {
        cy.get(`#resizable-modal .flags .dropdown-button.${flagName}`).should('have.length', 1);
        const editor = new InlineEditor('#resizable-modal .flag-comment');
        editor.value().should('contain', flagComment);
        return this;
    }

    isInputReadOnly(fieldName) {
        cy.get(`.task-details .modal-body .properties .form-group .control-label:contains("${fieldName}")`)
            .next()
            .find(`.field-readonly`)
            .should('exist');
        return this;
    }

    toggleFailureHandler() {
        const el = cy.get(".failure-handler").contains('Handling failure');
        el.click();
        // el.click();
        return this;
    }

    expectFailureHanderEditable() {
        cy.get(".failure-handler input#task-failure-script").should('not.have.attr', 'disabled');
    }

    expectFailureHanderNotEditable() {
        cy.get(".failure-handler div:contains('No failure handler')").should('be.visible');
    }

    enableFailureHandler() {
        cy.get('.failure-handler input[type="checkbox"]').should('be.not.checked');
        cy.get('.failure-handler input[type="checkbox"]').click();
        return this;
    }

    expectFailureHandlerHasDefaults() {
        cy.get('.failure-handler .bold:first').should('have.text', 'If the task fails');
        cy.get('.failure-handler button span').should('have.text', 'Skip task');
        return this;
    }

    expectFailureHandlerHas(isEnabled, action) {
        cy.get('.failure-handler input[type="checkbox"]').should(isEnabled ? 'be.checked' : 'be.not.checked');
        cy.get('.failure-handler button > .dropdown-label').should('have.text', action);
        return this;
    }

    expectFailureScriptHandlerIs(script) {
        const originalScript = new CodeInlineEditor('.failure-handler div[inline-jython-editor]').value();
        originalScript.should('contain', script);
        return this;
    }

    enterFailureHandlerScript(script) {
        cy.get('.failure-handler input[type="checkbox"]').check();
        cy.get('.failure-handler div[inline-jython-editor]').should('be.visible');
        new CodeInlineEditor('.failure-handler div[inline-jython-editor]').set(script);
        return this;
    }

    selectPostAction(action) {
        cy.wait(500);
        cy.get('.failure-handler .dropdown-selector').click();
        cy.get('.failure-handler .dropdown-selector').contains(action).click();
        cy.wait(500);
        return this;
    }

    togglePrecondition() {
        const precondition = cy.get(".precondition").contains('Precondition');
        precondition.scrollIntoView();
        precondition.click();
        return this;
    }

    expandFacetSection() {
        const el = cy.get('[data-test="facets"]').contains('Attributes');
        el.click();
        return new FacetSection();
    }

    expandPrecondition() {
        // it appears when tests runs on distribution server , task details modal loads
        // a lot of requests
        cy.wait(500);
        cy.get(".precondition:contains('Precondition')").then((element) => {
            if(!element.find('.precondition-form-group').length) {
                this.togglePrecondition();
            }
        });
        return this;
    }

    expectPreconditionEditable() {
        cy.get(".precondition div.task-precondition:visible");
        cy.get(".precondition pre").should('not.be.visible');
    }

    expectPreconditionNotEditable() {
        cy.get(".precondition div.task-precondition").should('not.be.visible');
        cy.get(".precondition pre").should('be.visible');
    }

}

class ParallelGroupDetails extends TaskDetails {
}

class GateDetails extends TaskDetails {

    countDependencies() {
        return element('#resizable-modal .dependency', 'Dependencies').count();
    }

    addDependency() {
        element('#resizable-modal .add-dependency', 'Add dependency link').click();
        return new Dependency("#resizable-modal .dependency:last");
    }

    getDependency(index) {
        const dependency = new Dependency(`#resizable-modal .dependency:nth('${index}')`);
        return dependency;
    }

    expectDependency(title) {
        expect(element(`.dependency-objects:contains('${title}')`).count()).toBe(1);
        return this;
    }
}
