import {CustomScriptDetails, GateDetails, ScriptDetails, TaskDetails} from '../components/task-details';
import Page from './page';
import ReleasePage from "./release-page";
import CalendarPage from "./calendar-page";

export default class TasksListPage {
    constructor() {
        cy.get('#tasks').should('be.visible');
    }

    clickOnFilter(criteria) {
        cy.get('.dropdown-button:contains("Filter options")').click();
        cy.get(`#filter-container label:contains('${criteria}')`).click();
        cy.get('.dropdown-button:contains("Filter options")').click();
        return this;
    }

    setSearchFilter(filterText) {
        cy.get('#searchFilter').clear().type(filterText);
        return this;
    }

    assignTaskToMe(title) {
        cy.get(`.task-line:contains('${title}') .assign-to-me`).click();
        cy.get(`.modal .modal-header`).should('contain', title);
        cy.get('.modal:visible .button:contains("Assign to me")').click();
        return this;
    }

    expectDisplayedReleasesCount(expectedCount) {
        cy.get(".release-name").should('have.length', expectedCount);
        return this;
    }

    expectTooManyResultsAlertDisplayed() {
        cy.get(".too-many-results-alert").should('have.length', 1);
        return this;
    }

    complete(title) {
        cy.get(`.task-line:contains('${title}') .complete:visible`).click();
        cy.get(`.modal:visible .modal-header`).should('contain', title);
        cy.get(`.modal:visible .modal-body textarea`).type('my relevant comment');
        cy.get(`.modal:visible .button:contains('Complete')`).click();
        cy.get(`.modal:visible .modal-header`).should('not.be.visible');
        cy.get(`.modal:visible .modal-body textarea`).should('not.be.visible');
        return this;
    }

    fail(title) {
        cy.get(`.task-line:contains('${title}') .fail:visible`).click();
        cy.get(`.modal:visible .modal-header`).should('contain', title);
        cy.get(`.modal:visible .modal-body textarea`).type('my relevant comment');
        cy.get(`.modal:visible .button:contains('Fail')`).click();
        cy.get(`.modal:visible .modal-header`).should('not.be.visible');
        cy.get(`.modal:visible .modal-body textarea`).should('not.be.visible');
        return this;
    }

    skip(title) {
        cy.get(`.task-line:contains('${title}') .skip:visible`).click();
        cy.get(`.modal:visible .modal-header`).should('contain', title);
        cy.get(`.modal:visible .modal-body textarea`).type('my relevant comment');
        cy.get(`.modal:visible .button:contains('Skip')`).click();
        cy.get(`.modal:visible .modal-header`).should('not.be.visible');
        cy.get(`.modal:visible .modal-body textarea`).should('not.be.visible');
        return this;
    }

    expectDateFilters(filters) {
        DateFilters.expectDateFilters(filters);
        return this;
    }

    expectTaskCountForRelease(releaseTitle, expectedCount) {
        cy.get(`#tasks-content div:contains('${releaseTitle}') .task`).should('have.length', expectedCount);
        return this;
    }

    expectTaskCompletable(title, completable) {
        cy.get(`.task-line:contains('${title}') .task-actions .complete:visible`).should('have.length', completable ? 1 : 0);
        return this;
    }

    expectTaskAssignedToMe(title) {
        return this.expectTaskAssignedTo(title, 'me');
    }

    expectTaskAssignedTo(title, user) {
        cy.get(`.task-line:contains('${title}') span.assignee span:contains('Assigned to'):visible span:contains('${user}'):visible`).should('have.length', 1);
        return this;
    }

    openManualTaskDetails(title) {
        return this.openTaskDetails(title, () => new TaskDetails());
    }

    openGateDetails(title) {
        return this.openTaskDetails(title, () => new GateDetails());
    }

    openNotificationDetails(title) {
        return this.openTaskDetails(title, () => new NotificationDetails());
    }

    openScriptDetails(title) {
        return this.openTaskDetails(title, () => new ScriptDetails());
    }

    openCustomScriptDetails(title) {
        return this.openTaskDetails(title, () => new CustomScriptDetails());
    }

    openTaskDetails(title, buildResult) {
        cy.get(`.task:contains('${title}') .task-line`).click();
        return buildResult();
    }

    refresh() {
        Page.softReload();
        return new TasksListPage();
    }

    expectTaskDisplayed(title) {
        cy.get(`.task-line:contains('${title}'):visible`, {timeout: 3000}).should('have.length', 1);
        return this;
    }

    expectTaskNotDisplayed(title) {
        cy.get(`.task-line:contains('${title}'):visible`, {timeout: 3000}).should('have.length', 0);
        return this;
    }

    expectModalClosed() {
        cy.get('.modal:hidden').should('have.length', 2);
        return this;
    }

    waitForTaskInProgress(title) {
        cy.get(`.task:contains('${title}') .status .labels span:contains('In progress')`)
            .should('be.visible', {timeout: 20});
        return this;
    }

    waitForTaskFailed(title) {
        cy.get(`.task:contains('${title}') .status .labels span:contains('Failed')`)
            .should('be.visible', {timeout: 20});
        return this;
    }

    expectReleaseDisplayed(title) {
        cy.get(`.release-name:contains('${title}')`).should('have.length', 1);
        return this;
    }

    expectTaskFlaggedWith(title, flagName) {
        cy.get(`.task:contains('${title}') .flag-line span.risk-flag-icon.${flagName}`).should('have.length', 1);
        return this;
    }

    expectTaskNotFlagged(title) {
        cy.get(`.task:contains('${title}') .flag-line:visible span.risk-flag-icon`).should('have.length', 0);
        return this;
    }

    toggleReleaseFold(title) {
        cy.get(`.task-header .release-name:contains('${title}')`).click();
        return this;
    }

    openReleaseOfTask(title) {
        cy.get(`.task:contains('${title}') a.view-in-release`).click();
        return new ReleasePage();
    }

    openCalendarViaTaskDueDate(title) {
        cy.get(`.task:contains('${title}') .calendar-link:contains('Due') a`).click();
        return new CalendarPage();
    }

    setOrderBy(orderBy) {
        cy.get('.order-selector .btn:first-child').click();
        cy.get(`.order-selector .bootstrap-link:contains('${orderBy}')`).click();
        cy.get('.order-selector .btn:first-child').click();
        return this;
    }

    // Pass in expected tasks per release which should be visible on position x...:
    // [
    //   {
    //      releaseTitle: "My first release",
    //      title: "My first task",
    //      position: 0,
    //   },
    //   {
    //      releaseTitle: "My first release",
    //      title: "My third task",
    //      position: 1, // second position because of ordering by due date
    //   }
    // ]
    expectTasksDisplayedInOrder(expectedTasks) {
        expectedTasks.forEach(task => {
            cy.get(`[data-release-title='${task.releaseTitle}'] .row .task`)
                .eq(task.taskPosition)
                .then(el => {
                    expect(el).to.be.visible;
                    expect(el).to.have.attr('data-task-title', task.taskTitle);
                });
        });
        return this;
    }

    expectTasksDisplayed(tasks) {
        Object.entries(tasks).forEach(([title, expected]) => {
            const expectedCount = Number(expected);
            cy.get(`.task:contains('${title}'):visible`).should('have.length', expectedCount);
        });
        return this;
    }
}
