import TagEditor from '../components/tag-editor';

export default class PermissionsPage {

    constructor() {
        cy.get('#permissions .table').should('be.visible');
    }

    uncheck(role, permission) {
        element(By.$(`tr:contains('${role}') .permission-${permission} .permission-checked:visible`)).click();
        return this;
    }

    expectChecked(role, permission) {
        expect(element.all(By.$(`tr:contains('${role}') .permission-${permission} .permission-checked:visible`)).count()).toEqual(1);
        return this;
    }

    add(role, permission) {
        const permissionEditor = new TagEditor(`tr:contains('${permission}')`);
        permissionEditor.addTag(`${role}`);
        return this;
    }

    save() {
        cy.get(".save").click();
        cy.wait(1000);
        return this;
    }


    reset() {
        cy.get(".reset").click();
        return this;
    }

    remove(role, permission) {
        cy.get(`tr:contains('${permission}') li:contains('${role}') .tag-close`).click();
        return this;
    }

    refresh() {
        cy.reload();
        return this;
    }

    expectPermissions(...definitions) {
        for (const definition of definitions) {

            Object.keys(definition.permissions).forEach((permission) => {
                let count = 0;
                if (definition.permissions[permission]) {
                    count = 1;
                }
                cy.get(`tr:contains('${permission}') td:contains('${definition.role}')`).should('have.length',count);
            });
        }
        return this;
    }

    expectResetEnabled(enabled = true) {
        if (enabled) {
            cy.get('.button.reset').should('have.length',1);
            // expect(element('.button.reset').count()).toBe(1);
            cy.get('.button.reset[disabled="disabled"]').should('have.length',0);
            // expect(element('.button.reset[disabled="disabled"]').count()).toBe(0);
        } else {
            cy.get('.button.reset[disabled="disabled"]').should('have.length',1);
            // expect(element('.button.reset[disabled="disabled"]').count()).toBe(1);
        }
        return this;
    }

    expectSaveEnabled(enabled = true) {
        if (enabled) {
            cy.get('.button.save').should('have.length',1);
            // expect(element('.button.reset').count()).toBe(1);
            cy.get('.button.save[disabled="disabled"]').should('have.length',0);
            // expect(element('.button.save').count()).toBe(1);
            // expect(element('.button.save[disabled="disabled"]').count()).toBe(0);
        } else {
            cy.get('.button.save[disabled="disabled"]').should('have.length',1);
            // expect(element('.button.save[disabled="disabled"]').count()).toBe(1);
        }
        return this;
    }

    expectSaveButtonInvisible() {
        expect(element.all(By.$(".save:visible")).count()).toEqual(0);
        return this;
    }

    addRoleToPermission(role, permission) {
        let lastSaved = '';
        cy
            .waitWithoutFailing('span.last-saved:visible')
            .then(() => {
                const el = Cypress.$('span.last-saved:visible');
                if (el && el.length) {
                    lastSaved = el.text();
                    // since we have no idea when previous save happened, and we do not want to be flaky,
                    // we want next save to have different "last save" time, so that we understand, that
                    // save happened. But save has only seconds, not milliseconds, thus let's wait for
                    // 2 seconds to make sure, that new time will be different.
                    // we need to wait couple of seconds to make last save time different to previous one
                    return new Promise((res) => {
                        setTimeout(res, 2000);
                    });
                }
            });
        const input = cy.get(`tr:contains('${permission}') .tag-input`).first();
        input.type(role);
        input.type('{enter}');
        cy.get(".save:visible").click()
            .waitWithoutFailing(() => {
                const el = Cypress.$('span.last-saved:visible');
                if (!el.length || el.text() === lastSaved) {
                    throw new Error('not saved yet');
                }
            })
    }

    addAdminPermission(role) {
        this.addRoleToPermission(role, 'Admin');
        return this;
    }

    addEditSecurityPermission(role) {
        this.addRoleToPermission(role, 'Edit security');
        return this;
    }

    addCreateTemplatePermission(role) {
        this.addRoleToPermission(role, 'Create template');
        return this;
    }

    addEditGlobalVariablesPermission(role) {
        this.addRoleToPermission(role, 'Edit global variables');
        return this;
    }

    addCreateTopLevelFoldersPermission(role) {
        this.addRoleToPermission(role, 'Create top level folders');
        return this;
    }

    addAndSaveAsCreateDashboard(role) {
        this.addRoleToPermission(role, 'Create dashboard');
        return this;
    }

    addAndSaveAsViewReports(role) {
        this.addRoleToPermission(role, 'View reports');
        return this;
    }
}

class ReleasePermissionsPage {
    constructor() {
        Browser.waitFor('#release-permissions #permissions-table');
    }

    expectSaveButtonDisabled() {
        expect(element.all(By.$(".save:disabled")).count()).toEqual(1);
        return this;
    }

    expectTeamMember(teamName, memberName, present) {
        expect(element(By.$(`#teams-permissions-table tr:contains('${teamName}')`))).toBePresent();
        expect(element(By.$(`#teams-permissions-table tr:contains('${teamName}'):contains('${memberName}')`))).toBePresent(present);
        return this;
    }

    expectPermissionGranted(permissionName, teamName, granted) {
        expect(element(By.$(`#permissions-table tr:contains('${permissionName}')`))).toBePresent();
        expect(element(By.$(`#permissions-table tr:contains('${permissionName}'):contains('${teamName}')`))).toBePresent(granted);
        return this;
    }

    expectInheritMessagePresent(folder) {
        expect(element(By.$("p")).getText()).toContain(`Permissions and Teams are inherited from the folder ${folder}`);
        return this;
    }
}
