import _ from 'lodash';

import {XlrDropdownWithVariables, XlrInlineTextWithVariables} from "../xlr-input-with-variables";

export class FacetSection {
    constructor() {
        this.root = '[data-test="facet-section"]';
        cy.get(this.root).should('have.length', 1);
    }

    addFacetClick() {
        cy.get(`${this.root} [data-test="add-facet"]`).click();
        cy.get('#facet-form-buttons .primary').should('be.disabled');
        return this;
    }

    selectFacetType(type) {
        cy.get('#facet-type .xlr-dropdown button').click();
        cy.get('#facet-type .xlr-dropdown [data-test="option"]').contains(type).click();
        cy.waitWithoutFailing(() => expect(Cypress.$('*:contains("{{")')).to.not.exist);
        cy.wait(100);
        return this;
    }

    submitFacetForm() {
        cy.get('#facet-form-buttons .primary').click();
        return this;
    }

    cancelFacetForm() {
        cy.get('#facet-form-buttons .cancel').click();
        return this;
    }

    expandNthFacetDetails(n) {
        cy.get('.facet-list .list-item').should('have.length.above', 0);
        cy.get('.facet-list .list-item').eq(n).click({force: true});
        return this;
    }

    fillFacetAutocompleteInputWithVariablesByName(name, value) {
        const xlWidgetAutocomplete = new XlrDropdownWithVariables(`${this.root} .facet-form div[id^="${name}"]`);
        xlWidgetAutocomplete.editor().focus().enter(value).select(value);
        return this;
    }

    fillFacetInlineTextInputWithVariablesByName(name, value) {
        const xlrInlineTextWithVariables = new XlrInlineTextWithVariables(`${this.root} .facet-form div[id^="${name}"]`);
        const editor = xlrInlineTextWithVariables.editor();
        editor.focus();
        editor.enter(value);
        return this;
    }

    deleteNthFacet(n) {
        cy.get('.facet-list .list-item').should('have.length.above', 0);
        cy.get('.facet-list .list-item .facet-actions [data-test="delete"]').eq(n).click();
        return this;
    }

    expectFacetFormInputToHaveValue(name, value) {
        cy.get(`${this.root} .facet-form [name="${name}"]`).should('have.value', value);
        return this;
    }

    expectFacetAutocompleteInputWithVariablesToHaveValue(name, value, isVariable = false) {
        const xlrInlineTextWithVariables = new XlrDropdownWithVariables(`${this.root} .facet-form div[id^="${name}"]`);
        xlrInlineTextWithVariables.editor().expectValue(value);
        return this;
    }

    expectFacetInlineTextInputWithVariablesToHaveValue(name, value, isVariable = false) {
        const xlrInlineTextWithVariables = new XlrInlineTextWithVariables(`${this.root} .facet-form div[id^="${name}"]`);
        const editor = xlrInlineTextWithVariables.editor();
        editor.value().should('to.contain', value);
        return this;
    }

    expectFacetsCountToBe(count) {
        cy.wait(800);
        cy.get('.facet-list .list-item').should('have.length', count);
        return this;
    }

    expectFacetListItemToBeReadonly(nth) {
        nth++;
        cy.get(`:nth-child(${nth}) .facet-list > .list-item #edit`).should('have.length', 0);
        cy.get(`:nth-child(${nth}) .facet-list > .list-item #delete`).should('have.length', 0);
        return this;
    }

    expectFacetToBeReadonly() {
        cy.get('#facet-type .xlr-dropdown button').should('be.disabled');
        cy.get('.editor > span').then(items => {
            _.each(items, item => {
                cy.wrap(item).should('have.class', 'field-readonly');
            });
        });
        return this;
    }
}
