import Modal from '../components/modal';
import ReleasePage from './release-page';
import ColorPicker from '../components/color-picker';
import ReleasesListPage from './releases-list-page';
import DateTimePicker from '../components/date-time-picker';

export default class CalendarPage {
    constructor() {
        cy.get('#calendar').should('be.visible');
    }

    reload() {
        return Page.softReload();
    }

    openDayModal(dayNumber) {
        cy.get(`.day:contains('${dayNumber}') .options-icon`).click();
        cy.get("xlr-context-menu a:contains('label')").click();
        return new CalendarDayModal();
    }

    setSearchFilter(filterText) {
        cy.get('#searchFilter').clear().type(filterText, {force: true});
        return this;
    }

    expectNavigatorHidden() {
        cy.get('#navigator:visible').should('have.length', 0);
        return this;
    }

    goToMonth(month) {
        cy.get(`#navigator .month-link:contains('${month}')`).first().click();
        return this;
    }

    expectMonthDisplayed(month) {
        cy.get(`.day span:contains('${month}'):visible`).should('have.length', 1);
        return this;
    }

    goToday() {
        cy.get('#calendar-header .show-today').click();
        return this;
    }

    showNextMonth() {
        const moment = Cypress.moment();
        const curMonth = moment.format('MMMM');
        cy.get('.button.previous-month').click(); // extra click to avoid an issue when UI is not re-rendered on next-month click
        cy.get('#current-month').then($el => {
            if ($el.text().indexOf(curMonth) === -1) { // check if UI was not responsive on first click: if was responsive and month is changed then extra click needed
                cy.get('.button.next-month').click();
            }
            cy.get('.button.next-month').click();
            const nextMonth = moment.add(1, 'months').format('MMMM');
            cy.get(`#current-month:contains('${nextMonth}')`).should('be.visible');
        });
        return this;
    }

    expectTodayDisplayed() {
        cy.get('.day.today').should('have.length', 1);
        return this;
    }

    expectDayTitle(dayNumber, title) {
        cy.get(`.day:contains('${dayNumber}'):contains('${title}')`).should('be.visible');
        return this;
    }

    goToNextMonth() {
        cy.get('.next-month').click();
        return this;
    }

    goToPreviousMonth() {
        cy.get('.previous-month').click();
        return this;
    }

    expectNavigatorDisplayed() {
        cy.get('#navigator:visible').should('have.length', 1);
        return this;
    }

    toggleNavigator() {
        cy.get('#navigator-toggler').click();
        return this;
    }

    clickOnFilter(criteria) {
        cy.get('button').click();
        cy.get(`#filter-container label:contains('${criteria}')`).click();
        return this;
    }

    expectTextFilter(text) {
        cy.get('#searchFilter').should('have.value', text);
        return this;
    }

    expectBooleanFilters(filters) {
        cy.get('button').click();
        for (const name in filters) {
            const expectedValue = filters[name];
            cy.get(`#display-${name}`).should('have.length', 1);
            cy.get(`#display-${name}:checked`).should('have.length', 0 + expectedValue);
        }
        cy.get('button').click();
        return this;
    }

    openReleaseTooltip(title) {
        cy.get(`.release .title:contains('${title}'):first`).should('have.length', 1);
        cy.get(`.release .title:contains('${title}'):first`).click();
        return new CalendarReleaseTooltip();
    }

    openReleasesForDay(dayNumber, month) {
        // {force:true} is the hour of the day hides the calendar date
        cy.get(".day.today").find('.day-number').click({force: true});
        cy.location('hash').should('contain', '/releases');
        return new ReleasesListPage();
    }

    searchBy(criteria) {
        //to handel waiting till all XHR requests are stopped loading
        cy.server();
        cy.route('/calendar/*').as('getCalendar');
        cy.route('/releases/*').as('getReleases');
        cy.get('#searchFilter').clear();

        cy.get('#searchFilter').type(criteria);

        cy.wait('@getCalendar');
        cy.wait('@getReleases');
        // let's wait until the calendar is redrawn
        cy.wait(1500);
        return this;
    }

    setBlackout(label, startDate, endDate) {
        this.openMenuItem('10', 'Set blackout period');
        const blackout = new CalendarBlackoutEditModal();
        blackout.setLabel(label);
        blackout.setStartDate(startDate);
        blackout.setEndDate(endDate);
        blackout.save();
        return this;
    }

    openMenuItem(dayNumber, item) {
        cy.wait(500); // somehow in a month of more then 2 same date values this test is flaky
        cy.get(`.day:contains('${dayNumber}') .options-icon`)
            .first()
            .click({force: true});
        cy.get(`xlr-context-menu a:contains('${item}')`).scrollIntoView().click();
    }

    expectBlackoutLabelToBe(label) {
        cy.get(`.blackout-label:contains("${label}")`).first().scrollIntoView().should('be.visible');
        return this;
    }

    openBlackoutTooltip(label) {
        cy.get(`.blackout-label:contains('${label}')`)
            .first()
            .scrollIntoView()
            .should('be.visible')
            .click({force: true});
        return new CalendarBlackoutTooltip();
    }

    expectReleasesDisplayed(releases) {
        return (() => {
            const result = [];
            for (const title in releases) {
                const expected = releases[title];
                if (expected) {
                    result.push(cy.get(`span:contains('${title}'):first`).should('have.length', 1));
                } else {
                    result.push(cy.get(`.release .release-title:contains('${title}'):visible`).should('have.length', 0));
                }
            }
            return result;
        })();
    }

    expectBlackoutToBePresent(label) {
        cy.get(`.blackout .blackout-label:contains("${label}")`).its('length').should('be.gt', 0);
        return this;
    }

    expectBlackoutToBeAbsent(label) {
        cy.get(`.blackout .blackout-label:contains("${label}")`).should('not.be.visible');
        return this;
    }

    // receives a list of colors because the color can change depending on whether the day is past or future
    expectDayColor(dayNumber, ...colors) {
        for (const color of colors) {
            cy.get(`.day:contains('${dayNumber}') .day-content`).should('have.css', 'background-color').and('eq', color);
        }
    }

    expectFirstDayOfWeekToBe(firstDayOfWeek) {
        cy.get(`#days-name > div:first-child:contains("${firstDayOfWeek}")`).should('be.visible');
        return this;
    }

    expectCurrentMonthToContain(title) {
        cy.get(`#current-month:contains('${title}')`).should('be.visible', {timeout: 20});
        return this;
    }
}

class CalendarBlackoutEditModal extends Modal {
    setLabel(text) {
        cy.get('.modal-dialog #label').clear();
        cy.get('.modal-dialog #label').type(text);
        return this;
    }

    setStartDate(date) {
        this.setDate('.start-date .date-editor', date);
        return this;
    }

    setEndDate(date) {
        this.setDate('.end-date .date-editor', date);
        return this;
    }

    setDate(selector, date) {
        const dateTimePicker = new DateTimePicker(selector);
        dateTimePicker.setDate({day: date.date(), month: date.month() + 1, year: date.year()});
        dateTimePicker.setTime({hour: date.hour(), minute: date.minute()}, () => cy.get('.modal-dialog #label').click());
    }

    expectTitleToBe(label) {
        cy.get('.modal-dialog .modal-header').should('contain', `Edit ${label}`);
        return this;
    }

    expectLabelToBe(label) {
        cy.get(`.modal-dialog input[ng-model="$ctrl.blackout.label"]`).should('have.value', label).should('have.length', 1);
        return this;
    }

    expectStartDateToBe(startDate) {
        cy.expectLocalDateValue('.start-date .date-editor .date input', startDate, 'shortDate');
        cy.expectLocalDateValue('.start-date .date-editor .time input', startDate, 'shortTime');
        return this;
    }

    expectEndDateToBe(endDate) {
        cy.expectLocalDateValue('.end-date .date-editor .date input', endDate, 'shortDate');
        cy.expectLocalDateValue('.end-date .date-editor .time input', endDate, 'shortTime');
        return this;
    }

    save() {
        cy.get('.modal-dialog .button.save').click();
        cy.get('.modal-content').should('not.be.visible');
    }

    cancel() {
        element(By.$('.modal-dialog .button.cancel')).click();
    }
}

class CalendarBlackoutDeleteModal extends Modal {
    del() {
        cy.get('.modal-dialog .button.continue').click();
        cy.get('.modal-content').should('not.be.visible');
    }
}

class CalendarBlackoutTooltip {
    constructor() {
        cy.get('.popover-blackout').should('be.visible');
    }

    openEdit() {
        cy.get('.link.edit').click({force: true});
        return new CalendarBlackoutEditModal();
    }

    openDelete() {
        cy.get('.link.delete').click({force: true});
        return new CalendarBlackoutDeleteModal();
    }

    close() {
        element(By.$('.popover-close')).click();
        return new CalendarPage();
    }

    expectLabelToBe(label) {
        cy.get('.popover-title > span').should('contain', label);
        return this;
    }

    expectStartDateToBe(startDate) {
        cy.expectLocalDateText('.popover-content .start-date', startDate, 'short');
        return this;
    }

    expectEndDateToBe(endDate) {
        cy.expectLocalDateText('.popover-content .end-date', endDate, 'short');
        return this;
    }
}

class CalendarReleaseTooltip {
    constructor() {
        cy.get(".calendar-tooltip").should('be.visible');
        cy.get(".calendar-tooltip:visible").should('have.length', 1);
    }

    openRelease() {
        cy.get(".calendar-tooltip:visible a:contains('View')").click();
        return new ReleasePage();
    }

    getOutgoingDependencies() {
        return element(".calendar-tooltip:visible .outgoing-dependencies").text();
    }

    getIncomingDependencies(text) {
        cy.get(".calendar-tooltip:visible .incoming-dependencies").should('contain', text);
        return this;
    }

    expectOutgoingDependencyFlagged(dependencyTitle) {
        cy.get(`.calendar-tooltip:visible .outgoing-dependencies:contains('${dependencyTitle}') .flagged`).should('have.length', 1);
    }

    expectIncomingDependencyFlagged(dependencyTitle) {
        cy.get(`.calendar-tooltip:visible .incoming-dependencies:contains('${dependencyTitle}') .flagged`).should('have.length', 1);
    }

    expectOutgoingDependencyNotFlagged(dependencyTitle) {
        cy.get(`.calendar-tooltip:visible .outgoing-dependencies:contains('${dependencyTitle}') .flagged`).should('have.length', 0);
    }
}

class CalendarDayModal {
    constructor() {
        cy.get(".modal-dialog").should('be.visible');
    }

    getColorPicker() {
        return new ColorPicker('.modal-dialog .pick-color');
    }

    setTitle(text) {
        cy.get('.input-block-level').clear();
        cy.get('.input-block-level').type(text);
        return this;
    }

    save() {
        cy.get('.modal-dialog .button.save').click();
        cy.get('.modal-dialog .button.save').should('not.be.visible');
        return this;
    }

    cancel() {
        cy.get('.modal-dialog .button.cancel').click();
        return this;
    }

    expectTitleToContain(title) {
        cy.get('.modal-header h4').should('contain', title);
        return this;
    }

    expectLabel(label) {
        cy.get('#day-label').should('contain', label);
        return this;
    }
}
