import TagEditor from '../components/tag-editor';

export const ids = {
    completedReleases: 'completed-releases',
    longestTaskType: 'longest-task-types',
    averageReleaseDuration: 'avg-release-duration',
    averageTaskDuration: 'avg-task-duration',
    taskEfficiency: 'task-efficiency',
    timeSpent: 'time-spent',
    releaseEfficiency: 'release-efficiency',
    numberOfReleases: 'number-of-releases',
    longestReleases: 'longest-releases',
    averageReleaseDurationOverTime: 'avg-release-duration-over-time',
    longestPhases: 'longest-phases',
    longestTasks: 'longest-tasks',
};

export const timeFrame = {
    last30Days: { id: 'LAST_MONTH', title: 'Last 30 days' },
    last3Months: { id: 'LAST_THREE_MONTHS', title: 'Last 3 months' },
    last6Months: { id: 'LAST_SIX_MONTHS', title: 'Last 6 months' },
    lastYear: { id: 'LAST_YEAR', title: 'Last year' },
    range: { id: 'RANGE', title: 'Range' },
};

export default class DashboardReportsPage {
    constructor() {
        cy.get('#report').should('be.visible');
    }

    addTags(...tags) {
        let tagEditor = new TagEditor('.tags');
        for (let tag of tags) {
            tagEditor.addTag(tag);
        }
        return this;
    }

    removeTags(...tags) {
        let tagEditor = new TagEditor('.tags');
        for (let tag of tags) {
            tagEditor.removeTag(tag);
        }
        return this;
    }

    selectTimeFrame(timeFrame) {
        cy.get(`#reports-header .actions .time-frame button:contains('${timeFrame.title}')`).click();
        return this;
    }

    generateTileTitlePath(id) {
        return `#${id} .card-title`;
    }

    expectCompletedReleasesTitle() {
        return this.expectTitleToBePresent(ids.completedReleases, 'Completed releases');
    }

    expectTitleToBePresent(id, title) {
        cy.get(`${this.generateTileTitlePath(id)} tile-title:contains('${title}')`)
            .scrollIntoView()
            .should('be.visible');
        return this;
    }

    expectMenuDisplayed() {
        cy.get(`navigation-sidebar h4:contains('Dashboards')`).should('be.visible');
        return this;
    }

    expectLongestTaskTypeTitle() {
        return this.expectTitleToBePresent(ids.longestTaskType, 'Longest task type');
    }

    expectAvgReleaseDurationTitle() {
        return this.expectTitleToBePresent(ids.averageReleaseDuration, 'Avg. release duration');
    }

    expectAvgTaskDurationTitle() {
        return this.expectTitleToBePresent(ids.averageTaskDuration, 'Avg. task duration');
    }

    expectTaskEfficiencyTitle() {
        return this.expectTitleToBePresent(ids.taskEfficiency, 'Task efficiency');
    }

    expectTimeSpentTitle() {
        return this.expectTitleToBePresent(ids.timeSpent, 'Time spent');
    }

    expectReleaseEfficiencyTitle() {
        return this.expectTitleToBePresent(ids.releaseEfficiency, 'Release efficiency');
    }

    expectNumberOfReleasesTitle() {
        return this.expectTitleToBePresent(ids.numberOfReleases, 'Number of releases');
    }

    expectTop10LongestReleasesTitle() {
        return this.expectTitleToBePresent(ids.longestReleases, 'Top 10 longest releases');
    }

    expectAverageReleaseDurationTitle() {
        return this.expectTitleToBePresent(ids.averageReleaseDurationOverTime, 'Average release duration over time');
    }

    expectTop10LongestPhasesTitle() {
        return this.expectTitleToBePresent(ids.longestPhases, 'Top 10 longest phases');
    }

    expectTop10LongestTasksTitle() {
        return this.expectTitleToBePresent(ids.longestTasks, 'Top 10 longest tasks');
    }

    expectTimeFrame(input) {
        cy.get(`#reports-header .actions .time-frame .active`).should('contain', input.title);
        cy.get(`#reports-header .dates .from-date .date-filter-editor .date-picker-holder`).should('contain', input.title);
        return this;
    }

    expectExportButton(id) {
        cy.get(this.generateExportButtonPath(id)).should('be.visible');
        return this;
    }

    generateExportButtonPath(id) {
        return `${this.generateTileTitlePath(id)} a:contains('Export')`;
    }
}
