// ***********************************************
// This example commands.js shows you how to
// create the custom command: 'login'.
//
// The commands.js file is a great place to
// modify existing commands and create custom
// commands for use throughout your tests.
//
// You can read more about custom commands here:
// https://on.cypress.io/api/commands
// ***********************************************
/* eslint-disable angular/json-functions */
import 'cypress-pipe';

import Errors from './dsl/components/errors';

import moment from 'moment';

const RUN_PID_FILE = './screenDiff.json';
const baseUrl = Cypress.env('baseUrl') ? Cypress.env('baseUrl') : 'http://localhost:5516';

const compareSnapshotCommand = require('cypress-image-diff-js/dist/command');
process.env['HEIGHT'] = '800';
process.env['WIDTH'] = '1280';
compareSnapshotCommand();

const angularJsDateFormatToMomentFormat = {
    'd/M/yy': 'D/M/YY',
    'M/d/yy': 'M/D/YY',
    'dd/MM/yy': 'DD/MM/YY',
    'yy/MM/dd': 'YY/MM/DD',
    'MMM d, y': 'MMM DD, YYYY',
    'dd MMM y': 'DD MMM YYYY',
    'y MMM dd': 'YYYY MMM DD',
};

const angularJsTimeFormatToMomentFormat = {
    'HH:mm': 'HH:mm',
    'h:mm a': 'h:mm A',
};

function _readRunIdFile() {
    return cy.readFile(RUN_PID_FILE);
}

Cypress.Commands.add(
    'login',
    (
        username,
        password,
        useAngularLocation = false,
        clearCookiesAndStorage = false,
        pathToOpen = '#/login',
        okSelector = '#releases-content',
        recoverySelector = 'top-toolbar .dot-branding a',
    ) => {
        const login = username || 'admin';
        const pass = password || 'admin';

        const log = Cypress.log({
            name: 'login',
            message: [login, pass],
            consoleProps: () => (login, pass),
        });

        if (useAngularLocation) {
            cy.navigate('/login');
        } else {
            cy.visit(pathToOpen);
        } // check if previous tests open the conditional modal for with saving state

        // Errors.clearErrors();

        cy
            // .waitUntilStable(500, () => cy.window().then(w => w.location.href))
            // .recoverFromForbiddenAccess('#login')
            .get('body')
            .then(($body) => {
                // synchronously query from body
                // to find which element was created
                if ($body.find(".button:contains('Leave')").length) {
                    // modal was found so closing it
                    cy.get(".button:contains('Leave')").click({ force: true });
                }
                $body.trigger('keydown', { key: 'Escape', code: 'Escape', which: 27 });
            })
            .get('#inputLogin')
            .click()
            .waitForAngular()
            .type(login)
            .get('#inputPassword')
            .waitForAngular()
            .click()
            .type(pass)
            .then(() => {
                if (clearCookiesAndStorage) {
                    cy.clearCookies();
                    cy.clearLocalStorage();
                    return cy;
                }
            })
            .get('.login-button button')
            .click()
            .waitUntilStable(1000, () => cy.window().then((w) => w.location.href))
            .doWithRetry(
                () => {},
                okSelector,
                () => cy.get(recoverySelector).click({ force: true }),
            )
            .get(okSelector)
            .should('be.visible')
            .location('hash')
            .should('not.eq', '#/login')
            .waitForAngular()
            .then(() => {
                if (okSelector === '#releases-content') {
                    return cy.waitWithoutFailing(
                        ['div.release-line:visible', 'div.alert.notice.releases-list-loaded:visible:contains(No releases found.)'],
                        2000,
                    );
                }
            })
            .then(() => {
                log.snapshot().end();
            });
    },
);

Cypress.Commands.add('logout', () => {
    Errors.clearErrors();
    cy.get('.avatar-wrapper button').click({ force: true });
    cy.get(".top-toolbar-avatar-menu li:contains('Log out')").click({ force: true });
    cy.waitWithoutFailing('.login-dialog');
    cy.recoverFromForbiddenAccess('#login');
    cy.get('.login-dialog').should('exist');
    Errors.clearErrors();
    // cy.wait(3000); // to let all the requests to complete
});

Cypress.Commands.add('recoverFromForbiddenAccess', (conditions) => {
    return cy
        .log('recoverFromForbiddenAccess')
        .waitUntilStable(500, () => cy.window().then((w) => w.location.href))
        .doWithRetry(
            () => {},
            conditions,
            () => cy.get('a.primary-logo').click({ force: true }),
        );
});

Cypress.Commands.add('deepClean', () => {
    cy.log('deepClean');
    cy.waitUntilStable(3000, () => cy.window().then((w) => w.location.href))
        .window()
        .then((w) => {
            try {
                w.onbeforeunload = null;
                w.location.href = `${baseUrl}/static/asdf/partials/login/login.html`;
            } catch (e) {}
            return cy.doWithRetry(
                () => {},
                () => expect(w.document.readyState).to.equal('complete'),
            );
        })
        .window()
        .then((w) => {
            w.sessionStorage.clear();
            w.localStorage.clear();
        })
        .then(() => {
            cy.clearCookies();
            cy.clearLocalStorage();
            return cy;
        });
});

Cypress.Commands.add('loginWithoutReload', (username, password, doNotRecover) => {
    cy.get('#inputLogin').clear().type(username);
    cy.get('#inputPassword').clear().type(password);
    cy.get('.login-button button').click();
});

Cypress.Commands.add('expectLocalDateValue', (selector, date, format) => {
    cy.get(selector)
        .then(($el) =>
            cy
                .window()
                .its('angular')
                .then((ng) => ng.element($el).injector()),
        )
        .then((injector) => {
            const dateFilter = injector.get('$filter')('date');
            const formattedDate = dateFilter(date, format);
            cy.get(selector).should('have.value', formattedDate);
        });
});

Cypress.Commands.add('expectLocalDateText', (selector, date, format) => {
    cy.get(selector)
        .then(($el) =>
            cy
                .window()
                .its('angular')
                .then((ng) => ng.element($el).injector()),
        )
        .then((injector) => {
            const dateFilter = injector.get('$filter')('date');
            const formattedDate = dateFilter(date, format);
            cy.get(selector).invoke('text').should('eq', formattedDate);
        });
});

Cypress.Commands.overwrite('visit', (orignalFn, url, options) => {
    cy.window().then((win) => {
        if (win.location.hash === url) {
            return cy.log(`visit0 ${url}`).stateReload().waitForAngular();
        } else {
            return cy
                .log(`visit1 ${baseUrl}/${url}`)
                .then(() => {
                    win.location = `${baseUrl}/${url}`;
                    return new Promise((res) => setTimeout(res, 50)).then(() =>
                        cy.doWithRetry(
                            () => {},
                            () => expect(win.document.readyState).to.equal('complete'),
                        ),
                    );
                })
                .waitForAngular();
        }
    });
});

Cypress.Commands.add('stopRequests', () => {
    cy.log('stopRequests')
        .window()
        .then((win) => {
            if (win.angular) {
                const injector = win.angular.element('body').injector();
                const RedirectOnLoginService = injector.get('RedirectOnLoginService');
                const ClientSettings = injector.get('ClientSettings');
                const redux = injector.get('$ngRedux');
                return cy.waitForAngular().then(() => {
                    redux.dispatch({ type: 'XLR_CLEAN_STATE' });
                    ClientSettings.clear();
                    RedirectOnLoginService.clean();
                    cy.clearLocalStorage();
                    return;
                });
            }
        });
});

// Cypress.Commands.overwrite('visit', (orignalFn, url, options) => {
//     return cy.wrap(orignalFn(url, options)).then(cy.waitForAngular);
// });

Cypress.Commands.overwrite('click', (orignalFn, subject, args) => {
    // don't wrap an already wrapped subject
    if (args && args.$el) {
        return orignalFn(subject, args);
    } else {
        return cy.wrap(orignalFn(subject, args)).waitForAngular();
    }
});

Cypress.Commands.add('compareToLocalDate', (timeStamp, locator, options) => {
    const epochTimeStamp = moment(timeStamp, 'DD-MM-YYYY').utc();
    const date = new Date(epochTimeStamp);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        return cy.log(`browser locale: ${locale}`).get(locator).should('have.value', date.toLocaleDateString(locale, options));
    });
});

Cypress.Commands.add('dateTimeFormat', () => {
    cy.window().then((win) => {
        if (win.angular) {
            const injector = win.angular.element('body').injector();
            const RegionalSettings = injector.get('RegionalSettings');
            const format = RegionalSettings.currentDateFormat + ' ' + RegionalSettings.currentTimeFormat;
            return cy.log('RegionalSettings dateTimeFormat: ', format).wrap(format);
        }
    });
});

Cypress.Commands.add('containLocalDate', (timeStamp, locator, options) => {
    const epochTimeStamp = moment(timeStamp, 'DD-MM-YYYY').utc();
    const date = new Date(epochTimeStamp);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        return cy.log(`browser locale: ${locale}`).get(locator).should('contain', date.toLocaleDateString(locale, options));
    });
});

Cypress.Commands.add('containLocalTime', (timeStamp, locator, options) => {
    const timeToCompare = moment(timeStamp, 'hh:mm');
    const date = new Date(timeToCompare);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        const formatedTime = date.toLocaleTimeString(locale, options);
        return cy.log('Browser locale', locale).get(locator).should('contain', formatedTime);
    });
});

Cypress.Commands.add('containLocalDateFromMilliseconds', (milliseconds, locator, options) => {
    const date = new Date(milliseconds);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        return cy.log(`browser locale: ${locale}`).get(locator).should('contain', date.toLocaleDateString(locale, options));
    });
});

Cypress.Commands.add('containLocalTimeFromMilliseconds', (milliseconds, locator, options) => {
    const date = new Date(milliseconds);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        const formatedTime = date.toLocaleTimeString(locale, options);
        return cy.log('Browser locale', locale).get(locator).should('contain', formatedTime);
    });
});

Cypress.Commands.add('waitForAngular', { prevSubject: 'optional' }, (elements) => {
    cy.log('waitForAngular');
    return cy
        .window()
        .then((win) => {
            return cy.doWithRetry(
                () => {
                    if (win.MYAPP) {
                        win.MYAPP.waitForAngular();
                    }
                },
                () => {
                    if (win.MYAPP) {
                        expect(win.MYAPP.APP_READY, 'MYAPP.APP_READY').to.be.true;
                    }
                },
            );
        })
        .then(() => elements);
});

Cypress.Commands.add('hint', { prevSubject: 'optional' }, (elements) => {
    if (!Cypress.config('isInteractive')) {
        // do nothing in non-interactive mode
        return Promise.resolve(elements);
    }
    return cy
        .log('hint')
        .window()
        .then((win) => {
            let go;
            const promise = new Promise((res) => {
                go = res;
            }).then((elements) => {
                const hints = win.document.getElementsByClassName('cypressHint');
                for (let i = hints.length - 1; i >= 0; i--) {
                    hints[i].parentNode.removeChild(hints[i]);
                }
                return elements;
            });
            let seenShift = false;
            const handler = (event) => {
                if (event.shiftKey) {
                    seenShift = true;
                } else if (seenShift && go) {
                    go(elements);
                    go = null;
                }
            };
            setTimeout(() => {
                if (go) {
                    go(elements);
                    go = null;
                }
            }, 29000); // user is sleeping, no need to wait further
            const append = (fn) => {
                const div = win.document.createElement('div');
                div.style.position = 'fixed';
                div.style.zIndex = 10000000;
                div.className = 'cypressHint';
                div.style.border = '3px solid red';
                fn(div);
                win.document.getElementsByTagName('body')[0].appendChild(div);
            };
            if (!elements || !elements.length) {
                append((div) => {
                    div.textContent = 'no elements found';
                });
            } else {
                elements.filter((index, element) => {
                    append((div) => {
                        const rect = element.getBoundingClientRect();
                        div.style.left = `${Math.floor(rect.left) - 5}px`;
                        div.style.width = `${Math.floor(rect.width) + 10}px`;
                        div.style.top = `${Math.floor(rect.top) - 5}px`;
                        div.style.height = `${Math.floor(rect.height) + 10}px`;
                    });
                });
            }
            append((div) => {
                div.zIndex = 10000001;
                div.style.left = '0px';
                div.style.top = '0px';
                div.style.width = '100%';
                div.style.height = '100%';
                div.style.border = '0px';
                div.addEventListener('mousemove', handler);
            });
            return promise;
        });
});

const wrapCondition = (condition) => {
    if ('function' === typeof condition) {
        return condition;
    } else if ('string' === typeof condition) {
        return () => expect(Cypress.$(condition)).to.exist;
    } else {
        throw new Error('Condition should be either a function or a selector string');
    }
};

Cypress.Commands.add('compareToLocalTime', (timeStamp, locator, options) => {
    const timeToCompare = moment(timeStamp, 'hh:mm');
    const date = new Date(timeToCompare);

    cy.window().then((win) => {
        const locale = win.navigator.language;
        const formatedTime = date.toLocaleTimeString(locale, options);
        return cy.get(locator).should('have.value', formatedTime);
    });
});

const diffPercentage = 0.2; // minimum allowed screen diff perentage of visual review server

Cypress.Commands.add('form_request', (method, url, formData) => {
    return cy
        .server()
        .route(method, url)
        .as('postImage')
        .window()
        .then((win) => {
            const xhr = new win.XMLHttpRequest();
            xhr.open(method, url);
            xhr.send(formData);
        })
        .wait('@postImage')
        .its('response')
        .then((resp) => {
            let status = true;
            return cy.readFile(RUN_PID_FILE).then((run) => {
                const url = `${Cypress.env('vrServer')}/api/runs/${run.body.id}/analysis`;
                return cy.request(url).then((analysis) => {
                    for (const item of analysis.body.diffs) {
                        if (item.after.imageId === resp.body.imageId) {
                            if (item.percentage >= diffPercentage && !Cypress.env('doBaseLine')) {
                                status = false;
                            }
                            expect(status, 'image comparison successful ?').to.be.true;
                        }
                    }
                });
            });
        });
});

Cypress.Commands.add('takeScreenShot', (fileName, options) => {
    if (Cypress.env('disableScreenDiff')) {
        return true;
    }
    cy.wait(1000);
    // take screenshot
    cy.screenshot(fileName, {
        blackout: options && options.blackout ? options.blackout : [],
        capture: options && options.capture ? options.capture : 'fullPage',
        clip: options && options.clip ? options.clip : null,
        disableTimersAndAnimations: options && options.disableTimersAndAnimations ? options.disableTimersAndAnimations : false,
        scale: options && options.scale ? options.scale : false,
        onBeforeScreenshot: options && options.onBeforeScreenshot ? options.onBeforeScreenshot : null,
        onAfterScreenshot($el, props) {
            Cypress.env('props', props);
        },
    }).then(() => {
        // get current run information
        return _readRunIdFile().then((run) => {
            const formData = new FormData();
            const method = 'POST';
            const url = `${Cypress.env('vrServer')}/api/runs/${run.body.id}/screenshots`;
            const fileType = 'image/png';
            const path = Cypress.env('props').path;
            const newPath = path.split(Cypress.config('screenshotsFolder'))[1];
            // get the file and send it to visual review server for image diff
            return cy.fixture(`../../../../../xlr-ui/build/cypress/screenshots/${newPath}`, 'base64').then((img) => {
                cy.task('log', `${Cypress.env('props').name} is generated`);
                return Cypress.Blob.base64StringToBlob(img, fileType).then((blob) => {
                    formData.set('file', blob, `${Cypress.env('props').name}.png`);
                    formData.set('screenshotName', Cypress.env('props').name);
                    formData.set(
                        'properties',
                        JSON.stringify({
                            resolution: Cypress.env('props').dimensions.width + 'x' + Cypress.env('props').dimensions.height,
                            browser: 'chrome',
                        }),
                    );
                    formData.set(
                        'compareSettings',
                        JSON.stringify({
                            precision: diffPercentage,
                        }),
                    );
                    formData.set('meta', JSON.stringify({}));
                    // make call and compare images
                    cy.form_request(method, url, formData);
                });
            });
        });
    });
});

Cypress.Commands.add('cleanErrors', () => {
    Errors.clearErrors();
});

const getAngular = () => cy.window().its('angular');

const loadApplicationIfNotLoaded = (url, closure) => {
    cy.hash().then((path) => {
        if (path === '' || !path.startsWith('#/')) {
            // application is not loaded
            return cy.visit(`#${url.startsWith('/') ? url : '/' + url}`);
        } else {
            return closure();
        }
    });
};

Cypress.Commands.add('navigate', (url) => {
    loadApplicationIfNotLoaded(url, () => {
        return cy.get('body').then(($el) =>
            getAngular().then((angular) => {
                const $injector = angular.element($el).injector();
                const $rootScope = $injector.get('$rootScope');
                const $location = $injector.get('$location');
                $location.url(url);
                $rootScope.$digest();
            }),
        );
    });
});

Cypress.Commands.add('stateReload', () => {
    loadApplicationIfNotLoaded('#/login', () => {
        return cy.get('body').then(($el) =>
            getAngular().then((angular) => {
                const $injector = angular.element($el).injector();
                const $rootScope = $injector.get('$rootScope');
                const $stateParams = $injector.get('$stateParams');
                const $state = $injector.get('$state');
                $state.transitionTo($state.current, $stateParams, {
                    reload: true,
                    inherit: false,
                    notify: true,
                    location: false,
                });
                $rootScope.$digest();
            }),
        );
    });
});

Cypress.Commands.add('waitForPopover', (buttonSelector) => {
    cy.doWithRetry(() => cy.get(buttonSelector).click({ force: true }), '.popover-content:visible');
});

Cypress.Commands.add('doWithRetry', (action, assertions, recover = null, timeout = 5000, retryFor = 30000) => {
    if (!(assertions instanceof Array)) {
        assertions = [assertions];
    }
    assertions = assertions.map(wrapCondition);
    let retries = 0;
    const retryUntil = new Date().getTime() + retryFor;
    let i;
    const attempt = () => {
        let actionResult;
        try {
            actionResult = retries && recover ? recover() : action();
        } catch (e) {
            return Promise.reject(e);
        }
        const next = () => {
            const waitUntil = new Date().getTime() + timeout;
            return new Promise((res, rej) => {
                function f() {
                    let exception;
                    for (i = 0; i < assertions.length; i++) {
                        try {
                            assertions[i]();
                            res();
                            return;
                        } catch (e) {
                            exception = e;
                        }
                    }
                    if (new Date().getTime() < waitUntil) {
                        setTimeout(f, 150);
                    } else {
                        retries++;
                        if (new Date().getTime() < retryUntil) {
                            cy.log('doWithRetry', `retry #${retries}`);
                            res(attempt());
                        } else {
                            rej(exception);
                        }
                    }
                }
                f();
            });
        };
        if (actionResult && 'function' === typeof actionResult.then) {
            return actionResult.then(next);
        } else {
            return Promise.resolve().then(next);
        }
    };
    return cy.wrap().then({ timeout: retryFor + 2 * timeout }, () => attempt());
});

Cypress.Commands.add('clickWithRetry', { prevSubject: true }, ($el, assertions) => {
    return cy.doWithRetry(() => $el.click(), assertions);
});

/**
 * The idea of this function is to give some grace period for page to stabilize, not too big (not 30 seconds).
 * Example usages are - we are waiting for tiles to appear within 2 seconds, but if they don't
 * that's also ok, we will consider, that there are no tiles to display. Thus, if tiles will appear
 * earlier - then we'll not wait for 2 seconds and will proceed faster.
 */
Cypress.Commands.add('waitWithoutFailing', (conditions, timeout = 5000) => {
    cy.log('waitWithoutFailing', conditions);
    if (!(conditions instanceof Array)) {
        conditions = [conditions];
    }
    conditions = conditions.map(wrapCondition);
    return cy.wrap().then(() => {
        let tries = 300;
        const when = timeout ? new Date().getTime() + timeout : 0;
        return new Promise((res, rej) => {
            function f() {
                let e;
                if (when && new Date().getTime() > when) {
                    res();
                    return;
                }
                let i;
                for (i = 0; i < conditions.length; i++) {
                    try {
                        conditions[i]();
                        res();
                        return;
                    } catch (e) {}
                }
                if (tries--) {
                    setTimeout(f, 150);
                } else {
                    rej(e);
                }
            }
            f();
        });
    });
});

Cypress.Commands.add('waitUntilStable', (timeout, conditions) => {
    if (!(conditions instanceof Array)) {
        conditions = [conditions];
    }
    const values = conditions.map((v) => null);
    let seenChange = new Date().getTime();
    let i = 0;
    const checkCondition = () => {
        return (
            cy
                // .log(`waitUntilStable[${i}, ${new Date().getTime()}]: ${new Date().getTime() - seenChange} vs ${timeout}`)
                .then(() => {
                    let res = null;
                    try {
                        res = conditions[i]();
                    } catch (e) {
                        cy.log(`Failed to execute condition: conditions[${i}]=${conditions[i]}`, e);
                    }
                    return res;
                })
                .then((value) => {
                    let outcome;
                    if (values[i] !== value) {
                        seenChange = new Date().getTime();
                        values[i] = value;
                        outcome = `waitUntilStable[${i}, ${new Date().getTime()}]: changed from ${values[i]} to ${value}`;
                    } else {
                        outcome = `waitUntilStable[${i}, ${new Date().getTime()}]: same ${value}`;
                    }
                    i++;
                    if (i === conditions.length) {
                        if (seenChange >= new Date().getTime() - timeout) {
                            i = 0;
                            return (
                                cy
                                    // .log(outcome)
                                    .then(() => new Promise((res) => setTimeout(res, 50)))
                                    .then(checkCondition)
                            );
                        } else {
                            return (
                                cy
                                    // .log(`${outcome}: completed`)
                                    .then(() => {})
                            );
                        }
                    } else {
                        return (
                            cy
                                // .log(outcome)
                                .then(checkCondition)
                        );
                    }
                })
        );
    };
    return checkCondition();
});

Cypress.Commands.add('clearFoldersCache', () => {
    return cy.window().then((win) => {
        win.angular.element('body').injector().get('FoldersService').clearCachedPromise();
    });
});

Cypress.Commands.add('toggleCollapse', (enabled) => {
    return cy.window().then((win) => {
        win.angular.element('body').injector().get('Collapse').enabled = enabled;
    });
});
