import { XlWidgetAutocomplete } from '../components/xl-components';

export default class ConfigurationPage {
    constructor() {
        cy.get('#configuration-header').should('be.visible');
    }

    refresh() {
        return new ConfigurationPage();
    }

    getInstance(instanceName) {
        return `.configuration-instance:contains('${instanceName}')`;
    }

    getType(typeName) {
        return `.configuration-type:contains('${typeName}')`;
    }

    openInstance(instanceName) {
        cy.get(`${this.getInstance(instanceName)} .edit-instance`).click({ force: true });
        return new ConfigurationInstancePage();
    }

    addNewInstance(typeName) {
        cy.get(`${this.getType(typeName)} .new-instance .button`).click({ force: true });
        return new ConfigurationInstancePage();
    }

    deleteInstance(instanceName) {
        cy.get(`${this.getInstance(instanceName)} .delete-instance`).click({ force: true });
        return this;
    }

    clearFilter() {
        cy.get('.search-input-filter input').clear({ force: true });
        return this;
    }

    searchInstances(criteria) {
        cy.window().then((win) => {
            win.document.getElementsByClassName('view-container')[0].getElementsByClassName('view')[0].scrollTo(0, 0);
        });
        cy.get('.search-input-filter input').click({ force: true });
        cy.get('.search-input-filter input').type(criteria, { force: true });
        return this;
    }

    expectTypeDisplayed(typeName) {
        expect(this.getType(typeName)).toBeDisplayed();
        return this;
    }

    expectInstanceDisplayed(instanceName) {
        cy.window().then((win) => {
            win.document.getElementsByClassName('view-container')[0].getElementsByClassName('view')[0].scrollTo(0, 100000);
        });
        cy.get(this.getInstance(instanceName)).should('have.length', 1);
        return this;
    }

    expectInstanceDisplayedOnContent(instanceName) {
        cy.get(this.getInstance(instanceName)).should('have.length', 1);
        return this;
    }

    expectInstanceNotDisplayed(instanceName) {
        cy.get(this.getInstance(instanceName)).should('have.length', 0);
        return this;
    }

    expectNoInstance(instanceName) {
        expect(this.getInstance(instanceName).isPresent()).toBe(false);
        return this;
    }

    expectErrorDisplayed() {
        cy.get('.modal').should('be.visible');
        cy.get('.modal .close').click();
        return this;
    }

    expectSuccessToasterToBeDisplayed(text) {
        cy.get('.toast-success').should('be.visible');
        cy.get('.toast-success').should('contain', text);
        return this;
    }
}

class ConfigurationInstancePage {
    constructor() {
        cy.get('#configuration-instance').should('exist');
    }

    setTextField(fieldName, fieldValue) {
        cy.get(`#${fieldName}`).clear({ force: true });
        cy.get(`#${fieldName}`).type(fieldValue, { force: true });
        return this;
    }

    // password-type fields in configurations use XlWidgetAutocomplete
    setPasswordField(fieldName, fieldValue) {
        cy.get(`#${fieldName} .display`).click();
        cy.get(`#${fieldName} .editable`).type(fieldValue).blur();
        return this;
    }

    togglePasswordFieldVariable(fieldName) {
        cy.get(`#${fieldName} .variable-toggle > button`).click({ force: true });
        return this;
    }

    setPasswordFieldVariable(fieldName, fieldValue) {
        const variablePicker = new XlWidgetAutocomplete(`#${fieldName}`);
        variablePicker.focus().select(fieldValue);
        return this;
    }

    setSelectField(fieldName, fieldValue) {
        cy.get(`div[name=${fieldName}]`).click();
        cy.get(`select[name=${fieldName}]`).select(fieldValue);
        return this;
    }

    expectFieldToBe(fieldName, fieldValue) {
        cy.get(`#${fieldName}`).should('have.value', fieldValue);
        return this;
    }

    expectFieldsToBeVisible(fieldNames) {
        fieldNames.forEach((fName) => {
            cy.get(`#${fName}`).should('be.visible');
        });
        return this;
    }

    expectFieldsToNotBeVisible(fieldNames) {
        fieldNames.forEach((fName) => {
            cy.get(`#${fName}`).should('not.be.visible');
        });
        return this;
    }

    expectFieldToNotContain(fieldName, fieldValue) {
        cy.get(`#${fieldName}`).should('not.have.value', fieldValue);
        return this;
    }

    expectPasswordVariableFieldToBe(fieldName, fieldValue) {
        const variablePicker = new XlWidgetAutocomplete(`#${fieldName}`);
        variablePicker.expectValue('${' + fieldValue + '}');
        return this;
    }

    save() {
        cy.get("action-toolbar button:contains('Save')").click();
        cy.get('.toast-success').should('be.visible');
        return this;
    }

    test() {
        cy.get("action-toolbar button:contains('Test')").click();
        return this;
    }

    expectTestButtonToBeDisabled() {
        cy.get("action-toolbar button:contains('Test')").should('be.disabled');
        return this;
    }

    selectFromDropdown(fieldName, element) {
        cy.get(`input[name=${fieldName}]`).clear({ force: true });
        cy.get(`input[name=${fieldName}]`).type(element + '{enter}', { force: true, delay: 100 });
        return this;
    }

    expectElementToContain(elementCss, value) {
        cy.get(elementCss).first().should('contain', value);
        return this;
    }

    expectSuccessToasterToBeDisplayed(text) {
        cy.get('.toast-success').should('be.visible');
        cy.get('.toast-success').should('contain', text);
        return this;
    }

    expectToBeSaved() {
        cy.get('.last-saved').should('be.visible');
        return this;
    }
}
