import Page from './page';
import {InlineEditor, TextareaInlineEditor, UserInlineEditor} from './inline-editor';
import ReleasesListPage from './releases-list-page';
import FolderTemplatesListPage from './folder-templates-list-page';
import ReleasePage from './release-page';
import TagEditor from '../components/tag-editor';
import {XlWidgetAutocomplete} from "../components/xl-components";

export class ReleasePropertiesPage {
    constructor() {
        cy.wait(1000);
        cy.get('#release-properties');
    }

    addTag(tagName) {
        return new TagEditor('#tags').addTag(tagName);
    }

    getTitle() {
        return input('$ctrl.releaseForm.title').val();
    }

    setTitle(title) {
        cy.get('#release-form-title')
            .clear()
            .type(title);
        return this;
    }

    getPageTitle() {
        return cy.get('.bread-crumbs span').text();
    }

    expectPageTitleToBe(releaseTitle) {
        cy.get('.bread-crumbs span').should('contain', releaseTitle);
        return this;
    }

    createAndExpectRedirectToTemplatesList() {
        cy.get('#release-properties .continue').click();
        return new FolderTemplatesListPage();
    }

    createAndExpectRedirectToReleasesList() {
        cy.get('#release-properties .continue').click();
        return new ReleasesListPage();
    }

    expectTemplateSelected(templateTitle) {
        cy.get(`.template-select input`).should('have.value', templateTitle);
        return this;
    }

    expectDuration(date){
        cy.get(`.duration.ng-binding:contains('${date}')`).should('have.length',1);
        return this;
    }

    expectDueDateEditable() {
        cy.get(`#release-properties .form-group:contains('Due date') .date-editor`)
            .should('have.length', 1);
        cy.get(`#release-properties .form-group:contains('Due date') .date-editor.readonly`)
            .should('have.length', 0);
        return this;
    }

    expectStartDateReadonly() {
        cy.get(`#release-properties .form-group:contains('Start date') .date-editor.readonly`)
            .should('have.length', 1);
        return this;
    }

    expectEndDateReadonly() {
        cy.get(`#release-properties .form-group:contains('End date') .date-editor.readonly`)
            .should('have.length', 1);
        return this;
    }

    expectReleaseTitleToBe(releaseTitle) {
        cy.get('input#release-form-title').should('have.value', releaseTitle);
        return this;
    }

    expectDescriptionToBe(releaseTitle) {
        new TextareaInlineEditor('.release-description')
            .value()
            .should('contain', releaseTitle);

        return this;
    }

    expectFlaggedWith(flagName, flagComment) {
        cy.get(`.flags .dropdown-button.${flagName}`).should('have.length', 1);
        cy.get('.flags input').should('have.value', flagComment);
        return this;
    }

    flag(flagName, flagComment) {
        cy.get('.flags .dropdown-button').click();
        cy.get('.flags .popover').should('have.length', 1);
        cy.get(`.flags .flag-picker-popover .${flagName}`).click();
        cy.get('.flags .dropdown-button').click();
        cy.get('.flags input').focus().type(flagComment);
        return this;
    }

    setOwner(owner) {
        const editor = new UserInlineEditor('.release-owner');
        editor.focus();
        editor.enter(owner);
        editor.blur();
        return this;
    }

    setAutomatedTaskUser(user) {
        cy.get('.script-username .display').click();
        cy.get('.script-username .edit .editable').type(user).blur();
        return this;
    }

    setAutomatedUserPass(pass) {
        cy.get('span[name="scriptUserPassword"] .display').click();
        cy.get('span[name="scriptUserPassword"] .editable').type(pass).blur();
        return this;
    }

    toggleAutomatedTaskUser() {
        cy.get('.user-input-with-variable .toggle-input-type button').click({force: true});
        return this;
    }

    selectAutomatedTaskUserVariable(username) {
        const variablePicker = new XlWidgetAutocomplete('.user-input-with-variable');
        variablePicker.focus().select('${'+username+'}');
        return this;
    }

    toggleAutomatedUserPass() {
        cy.get('.user-script-password .variable-toggle > button').click({force: true});
        return this;
    }

    selectAutomatedTaskPassVariable(password) {
        const variablePicker = new XlWidgetAutocomplete('.user-script-password');
        variablePicker.focus().select('${'+password+'}');
        return this;
    }

    expectAutomatedTaskUserToBe(value) {
        cy.log('check automated task user');
        // user-input-with-variable .variable-dropdown  input
        cy.get('.user-input-with-variable .variable-dropdown .xl-components.xl-widget-autocomplete > input').should('have.value', value);
        return this;
    }

    expectAutomatedTaskPassToBe(value) {
        cy.log('check automated task password');
        //  .user-script-password .variable-dropdown.ng-scope > div > div > input
        cy.get('.user-script-password .variable-dropdown .xl-components.xl-widget-autocomplete > input').should('have.value', value);
        return this;
    }

    expectOwnerToBe(owner) {
        (new UserInlineEditor('.release-owner').expectValue(owner));
        return this;
    }

    setPasswordVariable(variable, value) {
        const editor = new InlineEditor(`#release-properties .form-group:contains('${variable}') div[inline-password-editor]`);
        editor.set(value);
        return this;
    }

    setVariable(variable, value) {
        const editor = new InlineEditor(`#release-properties .form-group:contains('${variable}') div[inline-text-editor]`);
        editor.set(value);
        return this;
    }

    setValueProviderValue(key, value) {
        cy.get(`.inline-drop-down-list-editor>div[name="${key}"]`).click();
        cy.get(`.inline-drop-down-list-editor>div[name="${key}"] select[name="${key}"]`).select(value);
        return this;
    }

    getDescription() {
        return new TextareaInlineEditor('.release-description').value();
    }

    setDescription(description) {
        new TextareaInlineEditor('.release-description').set(description);
        return this;
    }

    removeTag(tagName) {
        return new TagEditor('#tags').removeTag(tagName);
    }

    expectReleaseTaggedWith(tagName) {
        new TagEditor('#tags').expectTaggedWith(tagName);
        return this;
    }

    expectReleaseNotTaggedWith(tagName) {
        new TagEditor('#tags').expectNotTaggedWith(tagName);
        return this;
    }

    expectVariablesCountToBe(variablesCount) {
        cy.get(`fieldset.variables`).should('have.length',variablesCount);
        return this;
    }

    save() {
        cy.get('#release-properties .continue').click();
        return this;
    }

    expectSaveDisabled() {
        cy.get(`#release-properties .continue:disabled`).should('have.length',1);
        return this;
    }

    expectSaveEnabled() {
        cy.get(`#release-properties .continue:enabled`).should('have.length',1);
        return this;
    }

    create() {
        cy.get('#release-properties .continue').click();
        cy.get('#release-header').should('be.visible');
        cy.wait(500);
        return new ReleasePage();
    }

    reload() {
        Page.softReload();
        return new ReleasePropertiesPage();
    }

    expectCalendarLinkContains(text) {
        cy.get('.subscribe-link').should('contain',text);
    }

    downloadICS() {
        cy.get('.download-ics').click();
    }

    expectSubscribeLinkIsPublic() {
        cy.get('#releaseForm-calendarPublished').scrollIntoView();
        cy.get('#releaseForm-calendarPublished').should('be.checked');
    }

    checkPublicLink() {
        cy.get('#releaseForm-calendarPublished').scrollIntoView();
        cy.get('#releaseForm-calendarPublished').check();
    }

    expectSubscribeLinkIsPrivate() {
        cy.get('#releaseForm-calendarPublished').scrollIntoView();
        cy.get('#releaseForm-calendarPublished').should('be.checked');
    }

    expectNoErrorMessage(message) {
        if (message !== null) {
            cy.get(`.error:contains('${message}'):visible`).should('have.length',0);
        } else {
            cy.get(`.error:visible`).should('have.length',0);
        }
        return this;
    }

    expectDescriptionToContain(description) {
        cy.get(".release-description.markdown").should('contain', description);
        return this;
    }

    selectTemplate(template) {
        cy.get('.template-select input').clear();
        if (template !== 'No template') {
            cy.get('.template-select input').type(template);
            cy.contains('.template-select .yt-option', template).click();
        }
        return this;
    }

    expectFolderToBe(folderTitle) {
        cy.get('xl-react-widget-tree-select .xl__single-value').should('contain', folderTitle);
        return this;
    }

    setFolder(folderId) {
        cy.wait(3000);
        cy.get('xl-react-widget-tree-select .xl__input input').click({force:true});
        cy.get(`xl-react-widget-tree-select .xl-react-widget-tree-item .xl-react-tree-node:contains('${folderId}')`).click();
        return this;
    }

    expectTemplateTitleToBe(title) {
       cy.get('a[data-test="created-from-template"]').invoke('text').should('eq', title);
        return this;
    }

    expectRiskProfileTitleToBe(title) {
        cy.get('a[data-test="release-risk-profile"]').invoke('text').should('eq', title);
        return this;
    }

    expectParentReleaseTitleToBe(title) {
        cy.get('.started-from-release').invoke('text').should('eq', title);
        return this;
    }

    expectTemplateURLToBe(url) {
        cy.get('a[data-test="created-from-template"]').should('have.attr', 'ng-href', url);
        return this;
    }

    expectParentReleaseURLToBe(url) {
        cy.get('.started-from-release').should('have.attr', 'ng-href', url);
        return this;
    }

}


export class TemplatePropertiesPage {
    constructor() {
        cy.get('.release-properties-form').should('be.visible');
    }

    addTag(tagName) {
        new TagEditor('#tags').addTag(tagName);
        return this;
    }

    setTitle(title) {
        cy.get('#release-form-title').type(title);
        return this;
    }

    create() {
        cy.get('.form-footer .button.continue').click();
        // cy.get('.new-release .button').should('be.visible');
        return this;
    }

    getDescription() {
        return new TextareaInlineEditor('.release-description').value();
    }

    setDescription(description) {
        return new TextareaInlineEditor('.release-description').set(description);
    }

    expectTemplateTaggedWith(tagName) {
        return new TagEditor('#tags').expectTaggedWith(tagName);
    }

    expectTemplateNotTaggedWith(tagName) {
        return new TagEditor('#tags').expectNotTaggedWith(tagName);
    }
}
