export default class TimelineCalendarPage {

    constructor() {
        cy.get('.timeline-page').should('be.visible');
    }

    openNavigationDrawer() {
        cy.get('.timeline-page .tl-header .tl-toggle-drawer-button').click();
        return new TimelineNavigationDrawer();
    }

    openDependencyDrawer(releaseTitle) {
        cy.get(`.tl-viewport .tl-event:contains("${releaseTitle}")`).click();
        cy.get(`.release-detail-popover .release-modal-header .release-modal-icon .cluster-dependencies-icon`).click();
        return new TimelineDependencyDrawer();
    }

    setReleaseTitleFilter(value) {
        cy.get('.timeline-page .tl-filter-basic .xl-react-widget-filter input').type(value);
        return this;
    }

    setReleaseTagsFilter(tags) {
        cy.get('.timeline-page .tl-filter-basic .tl-filter-release-tags .react-tagsinput').click();
        tags.forEach(tag => {
            cy.get(`.timeline-page .tl-filter-basic .tl-filter-release-tags div[title="${tag}"] i`).click({force: true});
        });
        return this;
    }

    setReleaseStatusFilter(statuses) {
        cy.get('.timeline-page .tl-filter-advanced .tl-filter-release-status .react-tagsinput').click();
        statuses.forEach(status => {
            cy.get(`.timeline-page .tl-filter-advanced .tl-filter-release-status .xl-react-autocomplete-suggestion div[title="${status}"] i.checkbox`).click({force: true});
        });
        return this;
    }

    setTag(tag) {
        let selectableTag = tag.toLowerCase();
        cy.get('.tl-filter-basic .tl-filter-release-tags').click();
        cy.get('.tl-filter-basic .tl-filter-release-tags .xl-react-link:contains("Clear all")').then($el => {
            if (!$el.is(':disabled')) {
                $el.click();
            }
        })
        cy.get('.tl-filter-basic .tl-filter-release-tags .with-search > input').clear().type(selectableTag);
        cy.get(`.tl-filter-basic .tl-filter-release-tags .xl-react-autocomplete-suggestion-list-container:nth-child(2) .xl-react-autocomplete-suggestion:contains("${selectableTag}")`).then($el => {
            if (!$el.is(':checked')) {
                $el.click();
            }
        });
        cy.get('.tl-filter-basic').click('topLeft', {force: true});
        return this;
    }

    toggleFilterFlaggedOnly() {
        cy.get('.timeline-page .tl-filter-advanced .tl-filter-flag-only').click();
        return this;
    }

    toggleFilterUserOnly() {
        cy.get('.timeline-page .tl-filter-advanced .tl-filter-user-only').click();
        return this;
    }

    toggleAdvancedFilters() {
        cy.get('.timeline-page .tl-filter-basic .tl-filter-advanced-toggle').click();
        return this;
    }

    expectFilterByReleaseTitlePresent() {
        cy.get('.timeline-page .tl-filter-basic .xl-react-widget-filter input').should('have.attr', 'placeholder', 'Filter by release title');
        return this;
    }

    expectFooterViewLabelToBe(date) {
        cy.get('.timeline-page .tl-footer-label').first().should('contain', date);
        return this;
    }

    expectTodayHoverToBeVisible() {
        cy.get('.timeline-page .tl-header-today').trigger('mouseover').should('be.visible');
        return this;
    }

    expectTodayHighlightedToBe(date) {
        cy.get('.timeline-page .tl-dates .tl-header-date.tl-today').should('contain', date);
        return this;
    }

    expectEventPresentWithTitle(title) {
        cy.get('.timeline-page .tl-content .tl-event .tl-event-title').should('contain', title);
        return this;
    }

    expectEventNotPresentWithTitle(title) {
        cy.get(`.timeline-page .tl-content .tl-event .tl-event-title:contains('${title}')`).should('not.exist');
        return this;
    }

    expectEventPresentWithFlag() {
        cy.get('.timeline-page .tl-content .tl-event .tl-event-info .release-status-label').should('contain', title);
        return this;
    }

    expectEventPresentWithStatus(status) {
        cy.get('.timeline-page .tl-content .tl-event .tl-event-info .release-status-label').should('contain', status);
        return this;
    }

    openBlackoutPopover(date) {
        return new BlackoutPeriod(date);
    }

}

class TimelineNavigationDrawer {

    navigateToMonth(date) {
        cy.get(`.tl-drawer-navigation-month__year:contains("${date.year()}") > .tl-drawer-navigation-month__months > .tl-drawer-navigation-month__name`)
            .eq(date.month())
            .click();
        return this;
    }

    expectNavigationDrawerToBeVisible() {
        cy.get('.tl-drawer-navigation').should('be.visible');
        return this;
    }

    expectHighlightedItemHasText(text) {
        cy.get('.tl-drawer-navigation-month__name.active').should('have.text', text);
        return this;
    }

    close() {
        cy.get('.tl-drawer-navigation > .tl-drawer-navigation-header .close-icon').click();
        return this;
    }
}

class TimelineDependencyDrawer {
    close() {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-header .close-icon`).click();
    }

    expectTitleToBe(title) {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-header h4`).should('contain', title);
        return this;
    }

    expectOutgoingDependencyTitleToBe(title) {
        cy.get('.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .tl-drawer-release-dependencies-group>span')
            .eq(1)
            .then($el => {
                expect($el).to.contain(title);
            });
        return this;
    }

    expectIncomingDependencyTitleToBe(title) {
        cy.get('.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .tl-drawer-release-dependencies-group>span')
            .eq(0)
            .then($el => {
                expect($el).to.contain(title);
            });
        return this;
    }

    expectToHaveIncomingDependencyOnRelease(releaseId) {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .incoming .tl-drawer-dependency[data-test-id="${releaseId}"]`)
            .should('exist');
        return this;
    }

    expectToHaveIncomingDependencyOnPhase(releaseId, phaseTitle) {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .incoming .tl-drawer-dependency[data-test-id="${releaseId}"] .tl-drawer-dependency-item-body-phase-title>span`)
            .should('contain', phaseTitle);
        return this;
    }

    expectToHaveIncomingDependencyOnTask(releaseId, taskTitle) {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .incoming .tl-drawer-dependency[data-test-id="${releaseId}"] .tl-drawer-dependency-item-body-task-title`)
            .should('contain', taskTitle);
        return this;
    }

    expectToHaveOutgoingDependencyOnRelease(releaseId) {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .outgoing .tl-drawer-dependency[data-test-id="${releaseId}"]`)
            .should('exist');
        return this;
    }

    expectToHaveOutgoingDependencyOnPhase(releaseId, phaseTitle) {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .outgoing .tl-drawer-dependency[data-test-id="${releaseId}"] .tl-drawer-dependency-item-body-phase-title>span`)
            .should('contain', phaseTitle);
        return this;
    }

    expectToHaveOutgoingDependencyOnTask(releaseId, taskTitle) {
        cy.get(`.tl-drawer-navigation .tl-drawer-navigation-body .tl-drawer-release-dependencies .outgoing .tl-drawer-dependency[data-test-id="${releaseId}"] .tl-drawer-dependency-item-body-task-title`)
            .should('contain', taskTitle);
        return this;
    }
}

class BlackoutPeriod {
    constructor(date) {
       this.date = date;
    }

    edit(title) {
        this.findBlackout(title);
        cy.get('.blackout-popover .blackout-popover-footer button:contains("Edit")').click();
        return this;
    }

    delete(title) {
        this.findBlackout(title);
        cy.get('.blackout-popover .blackout-popover-footer button:contains("Delete")').click({force: true});
        cy.get('.xl-react-widget-modal-dialog .dialog-buttons button.primary:contains("Delete")').click({force: true});
        return this;
    }

    findBlackout(title) {
        cy.get(`[date-test-date="${this.date}"]`).click({ force: true });
        cy.doWithRetry(
            () => cy.wait(50),
            () => expect(Cypress.$('div.blackout-popover-name').text()).to.equal(title),
            () => cy.get('.blackout-popover-navigation i.chevron-right-icon').click({force: true})
        );
    }

    setEndDate(date) {
        this.setDate(date, 'End date');
        return this;
    }

    save() {
        cy.get('.xl-react-modal-content .modal-footer button:contains("Save")').click();
        return this;
    }

    setDate(date, label) {
        const day = date.format('D');
        cy.get(`.xl-react-modal-content .modal-body .xl-react-components:has(.xl-react-components-label:contains("${label}")) .react-datepicker__input-container input`)
            .click({ force: true })
            .doWithRetry(
                () => {},
                () => expect(Cypress.$('div.react-datepicker__current-month:visible').text().trim().toLowerCase()).to.equal(date.format('MMMM YYYY').toLowerCase()),
                () => cy.get('button.react-datepicker__navigation.react-datepicker__navigation--next:visible').click({force: true}).wait(500),
                100
            )
            .get('div.react-datepicker__day:visible')
            .then((elements) => elements.filter((index, element) => !/react-datepicker__day--outside-month/.test(element.getAttribute('class')) && element.textContent.trim() === String(day))).then((e) => {
            if (e[0] && !/react-datepicker__day--selected/.test(e[0].getAttribute('class'))) {
                e[0].click();
            }
        });

        cy.get(`.xl-react-modal-content .modal-body .xl-react-components:has(.xl-react-components-label:contains("${label}")) .time-input input`)
            .click({force: true})
            .doWithRetry(() => {},
                () => expect(Cypress.$('div.popover-content:visible div.xl-react-widget-time-content:visible .hour-time:visible').text().trim()).to.equal(String(date.format('h'))),
                () => cy.get('button.plus-hour:visible').click({force: true}).wait(10),
                100)
            // minutes
            .doWithRetry(() => {},
                () => expect(Cypress.$('div.popover-content:visible div.xl-react-widget-time-content:visible .minute-time:visible').text().trim()).to.equal(String(date.format('mm'))),
                () => cy.get('button.plus-minute:visible').click({force: true}).wait(10),
                100)
            // AM/PM
            .doWithRetry(() => {},
                () => expect(Cypress.$('div.popover-content:visible div.xl-react-widget-time-content:visible div.am-pm:visible button:visible').text().trim()).to.equal(date.format('A')),
                () => cy.get('div.popover-content:visible div.xl-react-widget-time-content:visible div.am-pm:visible button:visible').click({force: true}).wait(10),
                100);
        cy.get('.xl-react-modal-content .modal-body').click('bottomLeft', {force: true});
    }

}
